import java.io.IOException as IOException
import sys
import os
import xldeploy
import requests
from urlparse import urlparse
from xldeploy.domain.Deployment import Deployment
from xlrxldeploy.ControlTaskRunner import ControlTaskRunner
from xldeploy.domain.QueryBuilder import QueryBuilder
from bs4 import BeautifulSoup
from xldeploy.errors import XLDeployException
from xldeploy.errors import XLDeployConnectionError
from xldeploy.errors import XLDeployConnectionTimeout
from xldeploy.errors import APIError
from xldeploy.domain.ConfigurationItem import ConfigurationItem
import tempfile
import urllib2
import ast
import json
import time
import traceback

from com.xebialabs.xlr.ssl import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption


class ObjectView(object):
    def __init__(self, d):
        self.__dict__ = d

    def getProperty(self, property):
        return self.__dict__[property]


def start_deployment(task, script_path, taskReportingApi):
    return _start_deployment(task, script_path, taskReportingApi)


def start_undeployment(task, script_path):
    return _start_undeployment(task, script_path)


def start_rollback(task, script_path):
    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    xld_task_id = task.getPythonScript().getProperty("xlDeployTaskId")
    xld_rollback_task = xld_exception_decorator(client.deployment.rollback, xld_task_id)

    task.setStatusLine("Deployment rollback triggered")
    task.schedule(script_path, 2)
    return xld_rollback_task.task_id


def cancel(task):
    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    xld_task_id = task.getPythonScript().getProperty("xlDeployTaskId")
    xld_exception_decorator(client.tasks.cancel, xld_task_id)


def _start_undeployment(task, script_path):
    try:
        fail_if_application_does_not_exist = task.getPythonScript().getProperty("failIfApplicationDoesNotExist")
        deployed_application = task.getPythonScript().getProperty("deployedApplication")
        if not check_ci_exist(task, deployed_application):
            if fail_if_application_does_not_exist:
                raise Exception("No deployed application found on {}".format(deployed_application))
            else:
                print("No deployed application found on {}".format(deployed_application))
                return
        xld_task_id = prepare_undeployment(task)
        task.setStatusLine("Starting undeployment")
        task.schedule(script_path, 2)
        return xld_task_id
    except:
        raise Exception("Undeployment of task: '%s' failed : [%s]\n" % (task, sys.exc_info()))


def _start_deployment(task, script_path, taskReportingApi):
    try:
        keepProp = task.keepPreviousOutputPropertiesOnRetry
        xld_task_id = task.getPythonScript().getProperty("xlDeployTaskId")
        if xld_task_id and keepProp:
            xld_task = check_deployment_status(task, xld_task_id)
            print "Task %s is in %s state.\n" % (xld_task_id, xld_task['state'])
            if xld_task['state'] in ('FAILED', 'ABORTED', 'STOPPED', 'CANCELLED'):
                task.setStatusLine("Restarting deployment")
                restart_task(task)
        else:
            xld_task_id = prepare_deployment(task, taskReportingApi)
            task.setStatusLine("Starting deployment")
        task.schedule(script_path, 2)
        return xld_task_id
    except:
        create_record(task, taskReportingApi, "failed", task.getPythonScript().getProperty("xlDeployTaskId"))
        raise Exception("Deployment of task: '%s' failed : [%s]\n" % (task, sys.exc_info()))


def check_status(task, script_path, server_url, connection_failures, process_deployment, delay=2):
    try:
        xld_task_id = task.getPythonScript().getProperty("xlDeployTaskId")
        task_with_steps = check_deployment_status(task, xld_task_id)

        if connection_failures > 0:
            connection_failures = 0
            print "Connection to XL Deploy '%s' is back.\n" % server_url
        task_state = task_with_steps['state']
        if task_state == "EXECUTING":
            current_and_total_steps = get_steps_count_from_xl_deploy(task.getPythonScript().getProperty("server"), task)
            task.setStatusLine(
                "Executing step %s/%s" % (
                    current_and_total_steps['currentStep'], current_and_total_steps['totalSteps']))
        task.schedule(script_path, delay)
        process_deployment(task_state)

    except Exception as e:
        connection_failures += 1
        connection_retries = task.getPythonScript().getProperty("connectionRetries")
        if connection_failures >= connection_retries:
            raise Exception("Failed to update task status after %s attempts: %s\n" % (connection_retries, e))
        else:
            print "Failed to connect to XL Deploy, will retry in 60 seconds: %s\n" % e
            task.schedule(script_path, 60)

    return connection_failures


def get_client_config(xld_server, task, username=None, password=None):
    xld_url = urlparse(xld_server.url)
    xld_protocol = xld_url.scheme
    xld_port = xld_url.port
    xld_host = xld_url.hostname
    overridden_username, overridden_password = None, None
    if task is not None:
        overridden_username = username if username else task.getPythonScript().getProperty("username")
        overridden_password = password if username else task.getPythonScript().getProperty("password")
    xld_username = overridden_username if overridden_username else xld_server.username
    xld_password = overridden_password if overridden_username else xld_server.password
    if xld_server.proxyHost and xld_server.proxyPort:
        config = xldeploy.Config(protocol=xld_protocol, host=xld_host, port=get_port(xld_port, xld_protocol), username=xld_username,
                                 password=xld_password, proxy_host=xld_server.proxyHost,
                                 proxy_port=xld_server.proxyPort, context_path=get_context_path(xld_url.path),
                                 proxy_username=xld_server.proxyUsername, proxy_password=xld_server.proxyPassword,
                                 verify_ssl=xld_server.getProperty('verifySSL'))
    else:
        config = xldeploy.Config(protocol=xld_protocol, host=xld_host, port=get_port(xld_port, xld_protocol), username=xld_username,
                                 password=xld_password, verify_ssl=xld_server.getProperty('verifySSL'),
                                 context_path=get_context_path(xld_url.path))
    return config


def get_api_client(xld_server, task=None, username=None, password=None):
    config = get_client_config(xld_server, task, username, password)
    return xldeploy.Client(config)


def query_xld_repository(security_api, configuration_api, task_api, request, params):
    xld_server = security_api.decrypt(configuration_api.getConfiguration(request.query['serverId']))
    task = security_api.decrypt(task_api.getTask('Applications/' + request.query['taskId'].replace('-', '/')))
    client = get_api_client(xld_server, task)
    try:
        result = client.repository.queryV3(params)
    except:
        result = []
    return result


def prepare_deployment(task, taskReportingApi):
    ensure_deployment_properties_filled(task)

    package_id = task.getPythonScript().getProperty("deploymentPackage")
    if not check_ci_exist(task, package_id):
        package_id = "Applications/" + package_id

    environment_id = task.getPythonScript().getProperty("deploymentEnvironment")
    if not check_ci_exist(task, environment_id):
        environment_id = "Environments/" + task.getPythonScript().getProperty("deploymentEnvironment")

    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    try:
        application_name = package_id.split("/")[-2]
        if not application_already_deployed(client, package_id, environment_id):
            deployment = client.deployment.prepare_initial(package_id, environment_id)
            print "Prepared Initial Deployment.\n"
        else:
            deployment = client.deployment.prepare_update(package_id, environment_id + '/' + application_name)
            print "Prepared Update Deployment.\n"

        deployment = client.deployment.prepare_auto_deployeds(deployment)
        deployment = enhance_deployment(deployment, task, True)
        valid_deployment = client.deployment.validate(deployment)

        if check_for_validation_messages(deployment) or check_for_validation_messages(valid_deployment):
            sys.exit(1)
        xld_task = client.deployment.create_task(deployment)
        create_record(task, taskReportingApi, "in_progress", xld_task.task_id)
        print "Created Deployment Task: %s.\n" % xld_task.task_id
    except Exception as e:
        catch_and_log_xld_exception(e)

    task.getPythonScript().setProperty("xlDeployTaskId", xld_task.task_id)
    return xld_task.task_id

def create_record(task, taskReportingApi, status, xld_task_id):
    try:
        record = taskReportingApi.newDeploymentRecord()
        record.targetId = task.id
        record.status = status
        xld_server = task.getPythonScript().getProperty("server")
        client_config = get_client_config(xld_server, task)

        record.serverUrl = xld_server.url
        record.serverUser = client_config.username

        record.deploymentTask = xld_task_id if xld_task_id else ''
        record.deploymentTask_url = (xld_server.url.rstrip("/") + "/#/task/" + record.deploymentTask) if xld_server.url and record.deploymentTask else ""

        package = task.getPythonScript().getProperty("deploymentPackage")
        version = task.getPythonScript().getProperty("deploymentVersion")
        if package and version and package.endswith(version):
            record.applicationName = package[:len(package) - len(version) - 1]
        else:
            record.applicationName = package
        record.environmentName = task.getPythonScript().getProperty("deploymentEnvironment")
        record.version = version
        taskReportingApi.addRecord(record, True)
    except Exception as e:
        print("Error while creating record: " + str(e))
        print(traceback.format_exc())

def prepare_undeployment(task):
    ensure_undeployment_properties_filled(task)

    deployed_application_id = task.getPythonScript().getProperty("deployedApplication")

    client = get_api_client(task.getPythonScript().getProperty("server"), task)

    deployment = client.deployment.prepare_undeploy(deployed_application_id)
    deployment = enhance_deployment(deployment, task, False)
    xld_task = client.deployment.create_task(deployment)
    print "Created undeployment Task: %s\n" % xld_task.task_id

    task.getPythonScript().setProperty("xlDeployTaskId", xld_task.task_id)
    return xld_task.task_id


def invoke_task_and_wait_for_undeployment(task, task_id, polling_interval=10, number_of_trials=None,number_of_continue_retrials=0):
    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    client.tasks.start(task_id)
    trial = 0
    while not number_of_trials or trial < number_of_trials:
        trial += 1
        status = client.tasks.get_task(task_id)['state']
        print 'Task [%s] is in [%s] state. \n' % (task_id, status)

        if status in ('FAILED', 'ABORTED', 'STOPPED') and number_of_continue_retrials > 0:
            status = invoke_task_and_wait_for_undeployment(task, task_id, polling_interval, number_of_trials,
                                                          number_of_continue_retrials - 1)
        elif status in ('FAILED', 'ABORTED', 'STOPPED', 'CANCELLED', 'DONE', 'EXECUTED'):
            for step in get_steps_log_from_xl_deploy(task, task_id):
                if step.log:
                    print "%s\n" % step.log.text
            break
        time.sleep(polling_interval)
    return status


def enhance_deployment(deployment, task, is_deployment):
    orchestrators = task.getPythonScript().getProperty("orchestrators")
    deployed_application_properties = task.getPythonScript().getProperty("deployedApplicationProperties")
    deployment = override_orchestrator(deployment, orchestrators)
    deployment = set_deployed_application_properties(deployment, deployed_application_properties)
    if is_deployment:
        deployment = set_deployed_properties(deployment, task.getPythonScript().getProperty("overrideDeployedProps"))
    return deployment

def override_orchestrator(deployment, orchestrators):
    if orchestrators:
        deployment_dict = deployment.to_dict()
        orchestrator_list = [orchestrator.strip() for orchestrator in orchestrators.split(",")]
        deployment_dict["application"]["orchestrator"] = orchestrator_list
        deployment = Deployment.as_deployment(deployment_dict)
    return deployment

def set_deployed_application_properties(deployment, deployed_application_properties):
    if deployed_application_properties:
        deployeds_application_properties_dict = dict(ast.literal_eval(deployed_application_properties))
        deployment_dict = deployment.to_dict()
        application = deployment_dict["application"]
        for key, value in deployeds_application_properties_dict.iteritems():
            application[key] = value
        deployment_dict["application"] = application
        deployment = Deployment.as_deployment(deployment_dict)
    return deployment

def set_deployed_properties(deployment, deployed_properties):
    deployment_dict = deployment.to_dict()
    if deployed_properties:
        for key, value in deployed_properties.iteritems():
            cis = __find_cis_for_deployable_id(deployment_dict['deployeds'], key)
            if len(cis) == 0:
                raise Exception("Deployable ID {} doesn't exist".format(key))
            else:
                for ci in cis:
                    __set_ci_properties(ci, json.loads(value))
    return Deployment.as_deployment(deployment_dict)

def __find_cis_for_deployable_id(deployeds, deployable_id):
    result = []
    for deployed in deployeds:
        if deployed['deployable'] == deployable_id :
            result.append(deployed)
    return result

def __set_ci_properties(ci, map):
    for updated_key, updated_value in map.iteritems():
        if isinstance(ci[updated_key], dict):
            if isinstance(updated_value, dict):
                for k, v in updated_value.iteritems():
                    ci[updated_key][k] = v
            else:
                raise Exception("CI property {} in id {} is map and not compatible with given property".format(updated_key, ci['deployable']))
        else:
            ci[updated_key] = updated_value

def ensure_deployment_properties_filled(task):
    if not task.getPythonScript().getProperty("server") or not task.getPythonScript().getProperty(
            "deploymentPackage") or not task.getPythonScript().getProperty("deploymentEnvironment"):
        raise Exception("Deployment package or environment is missing")


def ensure_undeployment_properties_filled(task):
    if not task.getPythonScript().getProperty("server") or not task.getPythonScript().getProperty(
            "deployedApplication"):
        raise Exception("Deployed application or server is missing")


def application_already_deployed(client, package_id, environment_id):
    application_name = package_id.split("/")[-2]
    query_params = QueryBuilder().type("udm.DeployedApplication").parent(environment_id).name_pattern(
        application_name).result_per_page(-1).build()
    application_list = xld_exception_decorator(client.repository.query, query_params)
    return application_list


def check_deployment_status(task, xld_task_id):
    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    xld_task = xld_exception_decorator(client.tasks.get_task, xld_task_id)
    start_or_archive_task(xld_task['state'], task, xld_task_id)
    return xld_task


def start_or_archive_task(task_state, task, xld_task_id):
    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    if task_state.lower() == "pending":
        xld_exception_decorator(client.tasks.start, xld_task_id)
        print "Started XL Deploy Task: %s\n" % xld_task_id
    elif task_state.lower() == "executed":
        xld_exception_decorator(client.tasks.archive, xld_task_id)
        print "Archived XL Deploy Task: %s\n" % xld_task_id


def get_steps_count_from_xl_deploy(server_id, task):
    xld_task_id = task.getPythonScript().getProperty("xlDeployTaskId")
    xld_config = get_client_config(server_id, task)
    path = "/task/%s/step" % xld_task_id
    proxies = None
    if xld_config.proxy_host and xld_config.proxy_port:
        proxies = get_proxies(xld_config)
    response = requests.get(get_xldeploy_url(xld_config, path=path), auth=(xld_config.username, xld_config.password),
                            headers=get_xml_headers(), proxies=proxies, verify=xld_config.verify_ssl)

    soup = BeautifulSoup(response.content)
    current_and_total_steps = {"currentStep": soup.task['currentstep'], "totalSteps": soup.task['totalsteps']}
    return current_and_total_steps


def get_steps_log_from_xl_deploy(task, task_id = None):
    xld_config = get_client_config(task.getPythonScript().getProperty("server"), task)
    path = "/task/%s/step" % (task_id if task_id else task.getPythonScript().getProperty("xlDeployTaskId"))
    proxies = None
    if xld_config.proxy_host and xld_config.proxy_port:
        proxies = get_proxies(xld_config)
    response = requests.get(get_xldeploy_url(xld_config, path=path), auth=(xld_config.username, xld_config.password),
                            headers=get_xml_headers(), proxies=proxies, verify=xld_config.verify_ssl)

    soup = BeautifulSoup(response.content)
    return soup.find_all('step')


def print_step_errors(task):
    steps = get_steps_log_from_xl_deploy(task)
    for step in steps:
        if int(step['failures']) > 0 and step.log:
            print "```\n%s\n```\n" % step.log.text


def get_proxies(config):
    proxies = None
    if config.proxy_host and config.proxy_port:
        if config.proxy_username and config.proxy_password:
            proxy_host_url = urlparse(config.proxy_host)
            proxy_username = urllib2.quote(config.proxy_username, safe='')
            proxy_password = urllib2.quote(config.proxy_password, safe='')

            proxy_url = "%s://%s:%s@%s:%s" % (proxy_host_url.scheme, proxy_username, proxy_password
                                              , proxy_host_url.hostname, config.proxy_port)
        else:
            proxy_url = "%s:%s" % (config.proxy_host, config.proxy_port)

        proxies = {'http': proxy_url,
                   'https': proxy_url}
    return proxies

def start_control_task(task, script_path):
    try:
        config = get_client_config(task.getPythonScript().getProperty("server"), task)

        control_task_runner = ControlTaskRunner(config)
        task_id = control_task_runner.execute_control_task(task.getPythonScript().getProperty("taskName"),
                                                           task.getPythonScript().getProperty("ciId"),
                                                           task.getPythonScript().getProperty("parameters"))
        task.getPythonScript().setProperty("xlDeployTaskId", task_id)
        task.setStatusLine("Starting control task")
        task.schedule(script_path, 2)
        return task_id
    except:
        raise Exception("Deployment of task: '%s' failed : [%s]\n" % (task, sys.exc_info()))

def check_for_validation_messages(deployment):
    validation_messages_exists = False
    for deployed in deployment.deployeds:
        if 'validation-messages' in deployed:
            print "Validation errors :\n"
            for message in deployed['validation-messages']:
                if "property" in message:
                    print "%s : %s, for property %s in CI %s.\n" % (
                        message.get('level'), message.get('message'), message.get('property'), message.get('ci'))
                else:
                    print "%s : %s, in CI %s.\n" % (
                        message.get('level'), message.get('message'), message.get('ci'))
                if message.get('level') == "ERROR":
                    validation_messages_exists = True
    return validation_messages_exists


def catch_and_log_xld_exception(e):
    if isinstance(e, XLDeployConnectionError):
        print "Connection error : %s.\n" % e.message
        raise e
    elif isinstance(e, XLDeployConnectionTimeout):
        print "`Connection to XL Deploy has been timed out.\n %s`" % e.message
        raise e
    elif isinstance(e, XLDeployException):
        message = e.explanation if e.explanation else e.message
        log_error_message(str(message))
        print "`Deployment Failed, please check the logs.\n`"
        sys.exit(1)
    else:
        raise e


def log_error_message(message):
    if 'Unauthorized for url:' in message:
        print "Authentication Error : Wrong credentials for XLD server.\n"
    elif 'Illegal name:' in message:
        print "Incorrect application name: %s.\n" % message[message.find(':') + 1:]
    elif 'Repository entity' in message and 'not found' in message:
        print "Invalid Package / Environment name: %s.\n" % message[message.find('[') + 1:message.find(']')]
    else:
        print "XL Deploy error: %s.\n" % message


def xld_exception_decorator(function, *args):
    try:
        return function(*args)
    except Exception as e:
        catch_and_log_xld_exception(e)


def set_ca_bundle_path():
    ca_bundle_path = extract_file_from_jar("requests/cacert.pem")
    os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path


def extract_file_from_jar(config_file):
    file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
    if file_url:
        tmp_file, tmp_abs_path = tempfile.mkstemp()
        tmp_file.close()
        Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
        return tmp_abs_path
    else:
        return None


if 'REQUESTS_CA_BUNDLE' not in os.environ:
    set_ca_bundle_path()


def get_xldeploy_url(config, path):
    return "%s://%s:%s/%s%s" % (config.protocol, config.host, config.port, config.context_path, path)


def get_xml_headers():
    return {'content-type': 'application/xml', 'Accept': 'application/xml'}


def check_ci_exist(task, path):
    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    return bool(client.repository.exists(path))


def get_json_headers():
    return {'content-type': 'application/json', 'Accept': 'application/json', 'Accept-Type': 'application/json'}


def display_step_logs(task):
    print "Printing step logs: "
    steps = get_steps_log_from_xl_deploy(task)
    for step in steps:
        if step.log:
            print "```\n%s\n```\n" % step.log.text


def restart_task(task):
    client = get_api_client(task.getPythonScript().getProperty("server"), task)
    xld_task_id = task.getPythonScript().getProperty("xlDeployTaskId")
    xld_exception_decorator(client.tasks.start, xld_task_id)
    print "Restarted XL Deploy Task: %s\n" % xld_task_id


def get_port(port, scheme):
    # if no port given in url, send http over port 80 and https over port 443, fallback to 80
    return port if port else 443 if scheme == 'https' else 80 if scheme == 'http' else 80


def get_context_path(context_path):
    return "{0}/deployit".format(context_path.strip("/")).lstrip("/") if context_path else 'deployit'


def get_deployed_app_on_env(xldeployServer, username, password, environment, date):
    data = {}
    server = {}
    for entry in xldeployServer.entrySet():
        server[entry.key] = entry.value
    server = ObjectView(server)
    xld_client = get_api_client(xld_server=server, username=username if username else server.username, password=password if password else server.password, task = {})

    if bool(xld_exception_decorator(xld_client.repository.exists, environment)):
        archived_tasks = xld_client.tasks.query(begin_date='2008-01-01', end_date=date)
        for task in archived_tasks:
            if task['state'] == 'DONE' and task['metadata']['taskType'] not in (
                    'CONTROL', 'INSPECTION', 'DEFAULT') and task['metadata']['environment_id'] == environment:
                if task['metadata']['taskType'] in ('INITIAL', 'UPGRADE', 'ROLLBACK'):
                    data[task['metadata']['application']] = get_row_data(task)
                if task['metadata']['taskType'] in 'UNDEPLOY' and task['metadata'][
                    'application'] in data:
                    del data[task['metadata']['application']]
    else:
        data = {"Invalid environment name"}

    return data



def get_row_data(task):
    row_map = {"id": task["id"], "application": task["metadata"]["application"],
               "version": task["metadata"]["version"], "owner": task["owner"], "date": task["completionDate"]}
    return row_map

