package com.xebialabs.xlrelease;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestContextManager;

import com.xebialabs.deployit.jcr.JcrTemplateHolder;
import com.xebialabs.deployit.plumbing.XLReleaseTest;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.xlrelease.db.ArchivedReleases;
import com.xebialabs.xlrelease.repository.ActivityLogs;
import com.xebialabs.xlrelease.rules.JcrTestInDirectoryRule;
import com.xebialabs.xlrelease.rules.JcrTestRule;

import static com.google.common.base.Throwables.propagate;
import static com.xebialabs.xlrelease.TestIds.RELEASES_DIR;
import static com.xebialabs.xlrelease.rules.JcrTestInDirectoryRule.newDirectory;
import static com.xebialabs.xlrelease.rules.LoginRule.grantAdminPermissionTo;
import static java.lang.Runtime.getRuntime;

/**
 * This is a base class for JUnit integration tests of XL Release. Inheritors of this class
 * can define {@link Autowired} XL Release services in fields and use them to setup a test
 * environment and assert conditions.
 * <p>
 *     A typical test would look like this:
 *
 * <pre>{@code
 * public class MyIntegrationTest extends XLReleaseIntegrationTest {
 *
 *     &#064;Autowired
 *     private RepositoryService repositoryService;
 *
 *     &#064;Test
 *     public void should_create_release_in_repository() {
 *         Release release = ReleaseBuilder.newRelease().withId(TestIds.RELEASE1).build();
 *
 *         repositoryService.create(release);
 *
 *         assertThat(repositoryService.read(TestIds.RELEASE1)).isNotNull();
 *     }
 * }
 * }</pre>
 * </p>
 * <p>
 *     <strong>Note:</strong> A test instance of XL Release is setup in a temporary folder and
 *     has some services mocked up for speed and easier testing. So you cannot test initializers or
 *     upgraders, for example.
 * </p>
 */
@RunWith(JUnit4.class)
@ContextConfiguration(locations = {"/spring/xlrelease-context-test.xml"})
public abstract class XLReleaseIntegrationTest extends XLReleaseTest {
    protected static JcrTestRule jcrTestRule;

    @Autowired
    private ArchivedReleases archivedReleases;

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private PermissionEditor permissionEditor;

    @Autowired
    private RoleService roleService;

    static {
        // Resorting to a static block, because a @BeforeClass method or a rule would
        // be triggered for each class
        jcrTestRule = new JcrTestRule();
        try {
            jcrTestRule.before();
        } catch (Exception e) {
            throw propagate(e);
        }

        XLReleaseIntegrationTestConfig.init(jcrTestRule);

        // Clean up the repository when testing completes
        // Using a shutdown hook so it's closed only once
        getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                jcrTestRule.after();
            }
        });
    }

    public XLReleaseIntegrationTest() {
        this(false);
    }

    public XLReleaseIntegrationTest(boolean useCache) {
        directory = newDirectory(RELEASES_DIR, jcrTestRule.getRepositoryService(), useCache);
    }

    @Before
    public void before() {
        try {
            // Autowire all fields
            new TestContextManager(getClass()).prepareTestInstance(this);
        } catch (Exception e) {
            throw propagate(e);
        }

        // Fix JcrTemplateHolder static init from other tests...
        new JcrTemplateHolder(jcrTestRule.getJcrTemplate());
        
        archivedReleases.init();

        grantAdminPermissionTo("admin", permissionEditor, roleService);
    }

    @After
    public void deleteLogsDirectory() {
        if (repositoryService.exists(ActivityLogs.ACTIVITY_LOGS_DIR)) {
            repositoryService.delete(ActivityLogs.ACTIVITY_LOGS_DIR);
        }
    }

    @Rule
    public JcrTestInDirectoryRule directory;
}
