package com.xebialabs.xlrelease;

import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.jcr.JcrTemplateHolder;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plumbing.XLReleaseTest;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.xlrelease.actors.ActorsInitializer;
import com.xebialabs.xlrelease.db.ArchivingDbInitializer;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.repository.ActivityLogs;
import com.xebialabs.xlrelease.rules.JcrTestInDirectoryRule;
import org.junit.After;
import org.junit.Before;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;
import org.python.google.common.base.Joiner;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestContextManager;

import java.util.Collections;
import java.util.List;

import static com.google.common.base.Throwables.propagate;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.TestIds.RELEASES_DIR;
import static com.xebialabs.xlrelease.rules.JcrTestInDirectoryRule.newDirectory;
import static com.xebialabs.xlrelease.rules.LoginRule.grantAdminPermissionTo;

/**
 * This is a base class for JUnit integration tests of XL Release. Inheritors of this class
 * can define {@link Autowired} XL Release services in fields and use them to setup a test
 * environment and assert conditions.
 * <p>
 * A typical test would look like this:
 * </p>
 * <pre>
 *     {@code
 * public class MyIntegrationTest extends XLReleaseIntegrationTest {
 *
 *     &#064;Autowired
 *     private RepositoryService repositoryService;
 *
 *     &#064;Test
 *     public void should_create_release_in_repository() {
 *         Release release = ReleaseBuilder.newRelease().withId(TestIds.RELEASE1).build();
 *
 *         repositoryService.create(release);
 *
 *         assertThat(repositoryService.read(TestIds.RELEASE1)).isNotNull();
 *     }
 * }
 * }
 * </pre>
 * <p>
 * <strong>Note:</strong> A test instance of XL Release is setup in a temporary folder and
 * has some services mocked up for speed and easier testing. So you cannot test initializers or
 * upgraders, for example.
 * </p>
 */
@RunWith(JUnit4.class)
@ContextConfiguration(locations = {"/spring/xlrelease-context-test.xml"})
public abstract class XLReleaseIntegrationTest extends XLReleaseTest {

    private final boolean useCache;
    /**
     * Some CIs can not be created inside the @{directory} which is deleted automatically. Those should be added to this collection to be purged after the test.
     */
    private List<String> cisForDeletion = newArrayList();

    @Autowired
    private ArchivingDbInitializer archivingDbInitializer;

    @Autowired
    public RepositoryService repositoryService;

    @Autowired
    public JcrTemplate jcrTemplate;

    @Autowired
    private PermissionEditor permissionEditor;

    @Autowired
    private RoleService roleService;

    @Autowired
    private ActorsInitializer actorsInitializer;

    public JcrTestInDirectoryRule directory;

    @Autowired
    protected ApplicationContext applicationContext;
    

    public XLReleaseIntegrationTest() {
        this(false);
    }

    public XLReleaseIntegrationTest(boolean useCache) {
        this.useCache = useCache;
    }

    @Before
    public void before() {
        try {
            // Autowire all fields
            new TestContextManager(getClass()).prepareTestInstance(this);
            directory = newDirectory(RELEASES_DIR, repositoryService, useCache);
            directory.before();
        } catch (Throwable throwable) {
            throw propagate(throwable);
        }

        // Fix JcrTemplateHolder static init from other tests...
        new JcrTemplateHolder(jcrTemplate);

        archivingDbInitializer.init();

        grantAdminPermissionTo("admin", permissionEditor, roleService);
    }

    @After
    public void tearDown() throws Exception{
        archivingDbInitializer.dropAll();
        directory.after();
        cleanActivityLogs();
        deleteCis();
        verifyRepositoryClean();
    }

    private void cleanActivityLogs() {
        if (repositoryService.exists(ActivityLogs.ACTIVITY_LOGS_DIR)) {
            repositoryService.delete(ActivityLogs.ACTIVITY_LOGS_DIR);
        }
    }

    private void verifyRepositoryClean() {
        final SearchParameters searchParameters = new SearchParameters();
        searchParameters.setParent("Applications");
        searchParameters.setType(Type.valueOf(Release.class));

        final List<ConfigurationItem> cis = repositoryService.listEntities(searchParameters);
        if (cis.size() > 0) {
            throw new RuntimeException("Found " + cis.size() + " releases after the test has finished: " + Joiner.on(",").join(cis));
        }

        //Clean up repository to avoid other tests failing
        deleteOnTearDown(cis.toArray(new ConfigurationItem[cis.size()]));
        deleteCis();
    }

    private void deleteCis() {
        for (String id : cisForDeletion) {
            repositoryService.delete(id);
        }

        cisForDeletion = newArrayList();
    }

    protected void deleteOnTearDown(ConfigurationItem... items) {
        for (ConfigurationItem item : items) {
            deleteOnTearDown(item.getId());
        }
    }

    protected void deleteOnTearDown(String... ids) {
        Collections.addAll(cisForDeletion, ids);
    }
}
