package com.xebialabs.xlrelease;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.junit.After;
import org.junit.Before;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;
import org.python.google.common.base.Joiner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestContextManager;

import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.jcr.JcrTemplateHolder;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plumbing.XLReleaseTest;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.xlrelease.actors.ActorSystemHolder;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.actors.ReleasesActorHolder;
import com.xebialabs.xlrelease.actors.utils.ReleaseActorLifecycleUtils;
import com.xebialabs.xlrelease.api.XLReleaseServiceHolder;
import com.xebialabs.xlrelease.api.v1.*;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.db.ArchivingDbInitializer;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.rules.JcrTestInDirectoryRule;

import scala.concurrent.duration.FiniteDuration;

import static com.google.common.base.Throwables.propagate;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.TestIds.RELEASES_DIR;
import static com.xebialabs.xlrelease.repository.ActivityLogs.ACTIVITY_LOGS_DIR;
import static com.xebialabs.xlrelease.rules.JcrTestInDirectoryRule.newDirectory;
import static com.xebialabs.xlrelease.rules.LoginRule.grantAdminPermissionTo;

/**
 * This is a base class for JUnit integration tests of XL Release. Inheritors of this class
 * can define {@link Autowired} XL Release services in fields and use them to setup a test
 * environment and assert conditions.
 * <p>
 * A typical test would look like this:
 * </p>
 * <pre>
 *     {@code
 * public class MyIntegrationTest extends XLReleaseIntegrationTest {
 *
 *     &#064;Autowired
 *     private RepositoryService repositoryService;
 *
 *     &#064;Test
 *     public void should_create_release_in_repository() {
 *         Release release = ReleaseBuilder.newRelease().withId(TestIds.RELEASE1).build();
 *
 *         repositoryService.create(release);
 *
 *         assertThat(repositoryService.read(TestIds.RELEASE1)).isNotNull();
 *     }
 * }
 * }
 * </pre>
 * <p>
 * <strong>Note:</strong> A test instance of XL Release is setup in a temporary folder and
 * has some services mocked up for speed and easier testing. So you cannot test initializers or
 * upgraders, for example.
 * </p>
 */
@RunWith(JUnit4.class)
@ContextConfiguration(locations = {"/spring/xlrelease-context-test.xml"})
public abstract class XLReleaseIntegrationTest extends XLReleaseTest {

    private static final Logger logger = LoggerFactory.getLogger(XLReleaseIntegrationTest.class);

    private final boolean useCache;
    /**
     * Some CIs can not be created inside the @{directory} which is deleted automatically. Those should be added to this collection to be purged after the test.
     */
    private List<String> cisForDeletion = newArrayList();

    @Autowired
    private ArchivingDbInitializer archivingDbInitializer;

    @Autowired
    public RepositoryService repositoryService;

    @Autowired
    public JcrTemplate jcrTemplate;

    @Autowired
    private PermissionEditor permissionEditor;

    @Autowired
    private RoleService roleService;

    @Autowired
    private ReleasesActorHolder releasesActorHolder;

    public JcrTestInDirectoryRule directory;

    @Autowired
    protected ApplicationContext applicationContext;

    @Autowired
    protected ReleaseActorService releaseActorService;

    @Autowired
    protected ActorSystemHolder actorSystemHolder;

    @Autowired
    private XlrConfig xlrConfig;

    @Autowired
    protected ReleaseActorLifecycleUtils releaseActorLifecycleUtils;

    public XLReleaseIntegrationTest() {
        this(false);
    }

    public XLReleaseIntegrationTest(boolean useCache) {
        this.useCache = useCache;
    }

    @Before
    public void before() {
        logger.debug("XLReleaseIntegrationTest 'before' started");
        try {
            // Autowire all fields
            new TestContextManager(getClass()).prepareTestInstance(this);
            directory = newDirectory(RELEASES_DIR, repositoryService, useCache);
            directory.before();
        } catch (Throwable throwable) {
            throw propagate(throwable);
        }

        // Fix JcrTemplateHolder static init from other tests...
        new JcrTemplateHolder(jcrTemplate);

        archivingDbInitializer.init();

        grantAdminPermissionTo("admin", permissionEditor, roleService);
        releasesActorHolder.awaitActorRef();

        XLReleaseServiceHolder.init(
                applicationContext.getBean(TaskApi.class),
                applicationContext.getBean(PhaseApi.class),
                applicationContext.getBean(ReleaseApi.class),
                applicationContext.getBean(TemplateApi.class),
                applicationContext.getBean(ConfigurationApi.class),
                applicationContext.getBean(RepositoryService.class)
        );

        // some actors may be initialized even from non-integration tests, e.g. via ReleaseExecutedEvent, so clean them up
        releaseActorLifecycleUtils.terminateAllReleaseActorsAndAwait(FiniteDuration.apply(5, TimeUnit.SECONDS));
        logger.debug("XLReleaseIntegrationTest 'before' finished");
    }

    @After
    public void tearDown() throws Exception {
        logger.debug("XLReleaseIntegrationTest 'after' started");
        releaseActorLifecycleUtils.terminateAllReleaseActorsAndAwait(FiniteDuration.apply(5, TimeUnit.SECONDS));
        archivingDbInitializer.dropAll();
        directory.after();
        cleanActivityLogs();
        deleteCis();
        verifyRepositoryClean();
        logger.debug("XLReleaseIntegrationTest 'after' finished");
    }

    private void cleanActivityLogs() {
        if (repositoryService.exists(ACTIVITY_LOGS_DIR)) {
            repositoryService.delete(ACTIVITY_LOGS_DIR);
        }
    }

    private List<Release> findAllReleases() {
        final SearchParameters searchParameters = new SearchParameters();
        searchParameters.setAncestor("Applications");
        searchParameters.setType(Type.valueOf(Release.class));
        return repositoryService.listEntities(searchParameters);
    }

    private void verifyRepositoryClean() {
        final List<Release> cis = findAllReleases();
        if (cis.size() > 0) {
            throw new RuntimeException("Found " + cis.size() + " releases after the test has finished: " + Joiner.on(",").join(cis));
        }
    }

    private void deleteCis() {
        for (String id : cisForDeletion) {
            repositoryService.delete(id);
        }

        cisForDeletion = newArrayList();
    }

    protected void deleteOnTearDown(ConfigurationItem... items) {
        for (ConfigurationItem item : items) {
            deleteOnTearDown(item.getId());
        }
    }

    protected void deleteOnTearDown(String... ids) {
        Collections.addAll(cisForDeletion, ids);
    }
}
