import React from 'react';
import PropTypes from 'prop-types';
import {XlReactWidgetReduxTable} from 'xl-react-components';
import {ButtonToolbar, DropdownButton} from 'react-bootstrap';
import {XlrMembersTags, XlrDeleteButton} from '..';
import './teams-table.less';
import {PageButton} from "./page-button";
import {PageSizeButton} from "./page-size-button";

const nameFilterClass = 'name-filter';

const initialTableState = {
    filters: [
        {
            className: nameFilterClass,
            fieldPath: 'teamName',
            key: 'teamName',
            label: 'Team name',
            model: '',
            placeholder: '',
            position: {
                column: 0,
                row: 0
            },
            type: 'input'
        },
        {
            className: nameFilterClass,
            fieldPath: 'roles',
            key: 'roles',
            label: 'Global role',
            model: '',
            placeholder: '',
            position: {
                column: 0,
                row: 0
            },
            type: 'input'
        },
        {
            className: nameFilterClass,
            fieldPath: 'users',
            key: 'users',
            label: 'Users',
            model: '',
            placeholder: '',
            position: {
                column: 0,
                row: 0
            },
            type: 'input'
        }
    ],
    layout: [
        {id: 0, dataKey: "teamName", title: "Team name"},
        {id: 1, dataKey: "roles", title: "Global role"},
        {id: 2, dataKey: "users", title: "Users"},
    ],
    paging: {
        page: 1,
        pageSizes: [
            {text: '5', value: 5},
            {text: '10', value: 10},
            {text: '20', value: 20},
            {text: '50', value: 50},
            {text: '100', value: 100},
        ],
        resultsPerPage: 10
    }
};

export class TeamsTable extends React.Component {

    static propTypes = {
        initTables: PropTypes.func.isRequired,
        loadedTableId: PropTypes.any, //dirty hack to not try to apply mapStateToProps before init
        onAddRole: PropTypes.func.isRequired,
        onAddUser: PropTypes.func.isRequired,
        onRemoveRole: PropTypes.func.isRequired,
        onRemoveTeam: PropTypes.func.isRequired,
        onRemoveUser: PropTypes.func.isRequired,
        paginateDataApi: PropTypes.func.isRequired,
        readOnly: PropTypes.bool,
        roles: PropTypes.arrayOf(PropTypes.shape({
            name: PropTypes.string.isRequired,
            type: PropTypes.string,
        })),
        tableId: PropTypes.string.isRequired,
        users: PropTypes.arrayOf(PropTypes.shape({
            name: PropTypes.string.isRequired,
            fullName: PropTypes.string,
        })),
    };

    static defaultProps = {
        readOnly: false,
    };

    componentDidMount() {
        const {tableId} = this.props;
        this.props.initTables({
            ...initialTableState,
            paging: {
                ...initialTableState.paging,
                pageButtonRenderer: this.pageButtonRenderer,
                sizePerPageRenderer: this.sizePerPageRenderer,
            },
            tableId,
        });
    }

    pageButtonRenderer = (props) => <PageButton key={props.page} {...props} />

    sizePerPageRenderer = ({options, currSizePerPage, onSizePerPageChange}) =>
        <ButtonToolbar className="react-bs-table-sizePerPage-dropdown user-is-tabbing">
            <DropdownButton dropup={true} id="pagination-size-per-page" title={currSizePerPage}>
                {options.map((option) =>
                    <PageSizeButton key={option.text} onClick={() => onSizePerPageChange(option.page)} option={option} />
                )}
            </DropdownButton>
        </ButtonToolbar>
    ;

    formatCell = (dataKey) => (cell, row) => {
        switch (dataKey) {
            case 'teamName':
                return cell;
            case 'roles':
                return this.formatRoleCell(cell, row);
            case 'users':
                return this.formatUserCell(cell, row);
            default:
                return cell;
        }
    }

    formatRoleCell = (cell, row) =>
        <XlrMembersTags
            allowAddOnlyFromSuggestion={true}
            classNames={{tagInput: 'tag-new role', selected: 'tag role'}}
            handleAddition={this.addRoleToTeam(row)}
            handleDelete={this.removeRoleFromTeam(row)}
            label='displayName'
            readOnly={this.props.readOnly || row.readOnly}
            suggestions={this.mapRoles(row, this.props.roles)}
            tags={this.mapRoles(row, cell)}
        />;

    formatUserCell = (cell, row) =>
        <div className={'actions'}>
            <div className={'member-tags'}>
                <XlrMembersTags
                    handleAddition={this.addUserToTeam(row)}
                    handleDelete={this.removeUserFromTeam(row)}
                    label='displayName'
                    readOnly={this.props.readOnly || row.readOnly}
                    suggestions={this.mapUsers(row, this.props.users)}
                    tags={this.mapUsers(row, cell)}
                />
            </div>
            {(!this.props.readOnly && !row.readOnly) && !row.systemTeam && <XlrDeleteButton
                item={row}
                modalHeader={'Delete Team'}
                onDelete={this.deleteTeam}
                renderModalBodyFn={() => this.renderDeleteModal(row)}
            />}
        </div>

    addUserToTeam = (team) => (user) => this.props.onAddUser(team, user);

    removeUserFromTeam = (team) => (index) => this.props.onRemoveUser(team, index);

    addRoleToTeam = (role) => (user) => this.props.onAddRole(role, user);

    removeRoleFromTeam = (role) => (index) => this.props.onRemoveRole(role, index);

    mapUsers = (team, suggestions) => suggestions.map(x => ({
                ...x,
                id: x.name,
                displayName: x.fullName ? `${x.fullName} (${x.name})` : x.name,
                readOnly: team.readOnly || this.props.readOnly,
        })
    );

    mapRoles = (team, suggestions) => suggestions.map(x => ({
            ...x,
            id: x.name,
            displayName: x.name,
            readOnly: team.readOnly || this.props.readOnly,
        })
    );


    renderDeleteModal = (team) => <div>You are about to delete the team <strong>{team.teamName}</strong></div>

    deleteTeam = (team) => this.props.onRemoveTeam(team)

    createColumnDefinition = (headerDef) => ({
        dataField: headerDef.dataKey,
        sort: false,
        text: headerDef.title,
        formatter: this.formatCell(headerDef.dataKey)
    });

    render() {
        const {loadedTableId, tableId, users, paginateDataApi} = this.props;
        return (
            <React.Fragment>
                {loadedTableId && users.length && <XlReactWidgetReduxTable
                    className={"transparent"}
                    createColumnDefinition={this.createColumnDefinition}
                    defaultErrorMessage={"Something wrong happened"}
                    defaultNoDataText={"No teams found"}
                    displayOnlyLabelIfNoData={false}
                    fetchDataApi={paginateDataApi}
                    filterType="column"
                    keyField="id"
                    loadingMessage={"loading..."}
                    tableId={tableId}
                />}
            </React.Fragment>
        );
    }

}
