package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.forms.*;
import com.xebialabs.xlrelease.domain.delivery.Delivery;
import com.xebialabs.xlrelease.domain.delivery.Stage;
import com.xebialabs.xlrelease.domain.delivery.TrackedItem;
import com.xebialabs.xlrelease.domain.delivery.Transition;

/**
 * Operations on release deliveries.
 */
@Path("/api/v1/delivery-patterns")
@Produces({MediaType.APPLICATION_JSON})
@Consumes({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface DeliveryPatternApi extends ApiService {

    default String serviceName() {
        return "deliveryPatternApi";
    }

    /**
     * Creates a new delivery pattern.
     *
     * @param pattern the delivery pattern to be created.
     * @return the created delivery pattern.
     */
    @POST
    @PublicApiMember
    Delivery createPattern(Delivery pattern);

    /**
     * Returns a delivery pattern by ID.
     *
     * @param patternId the pattern identifier.
     * @return the pattern which has the given identifier.
     */
    @GET
    @Path("/{patternId:.*Delivery[^/]*}")
    @PublicApiMember
    Delivery getPattern(@PathParam("patternId") String patternId);

    /**
     * Returns a delivery pattern by ID or title.
     *
     * @param patternIdOrTitle the pattern ID or title.
     * @return the pattern which has the given ID or the given title.
     */
    @GET
    @Path("/{patternIdOrTitle}")
    @PublicApiMember
    Delivery getPatternByIdOrTitle(@PathParam("patternIdOrTitle") String patternIdOrTitle);

    @POST
    @Path("/checkTitle")
    boolean checkTitleUnique(ValidatePattern validation);

    /**
     * Updates a delivery pattern.
     *
     * @param patternId the pattern identifier.
     * @param pattern   new contents of the pattern.
     * @return the updated pattern.
     */
    @PUT
    @Path("/{patternId:.*Delivery[^/]*}")
    @PublicApiMember
    Delivery updatePattern(@PathParam("patternId") String patternId, Delivery pattern);

    /**
     * Updates a delivery pattern.
     *
     * @param pattern new contents of the pattern.
     * @return the updated pattern.
     */
    @PublicApiMember
    Delivery updatePattern(Delivery pattern);

    /**
     * Deletes a delivery pattern.
     *
     * @param patternId the pattern identifier.
     */
    @DELETE
    @Path("/{patternId:.*Delivery[^/]*}")
    @PublicApiMember
    void deletePattern(@PathParam("patternId") String patternId);

    @POST
    @Path("/{patternId:.*Delivery[^/]*}/duplicate")
    Delivery duplicatePattern(@PathParam("patternId") String patternId, DuplicateDeliveryPattern duplicateDeliveryPattern);

    /**
     * Creates a delivery from a delivery pattern.
     *
     * @param patternId      the identifier of the delivery pattern the delivery is based on.
     * @param createDelivery the parameters for the new delivery.
     * @return the newly created delivery.
     */
    @POST
    @Path("/{patternId:.*Delivery[^/]*}/create")
    @PublicApiMember
    Delivery createDeliveryFromPattern(@PathParam("patternId") String patternId, CreateDelivery createDelivery);

    /**
     * Search delivery patterns by filters.
     *
     * @param filters        the search criteria.
     * @param page           the page of results to return. Default value is 0.
     * @param resultsPerPage the number of results per page. Default and maximum value is 100.
     * @return the list of matching delivery patterns.
     */
    @POST
    @Path("search")
    @PublicApiMember
    List<Delivery> searchPatterns(DeliveryPatternFilters filters,
                                  @DefaultValue("0") @QueryParam(PAGE) Long page,
                                  @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) Long resultsPerPage);

    /**
     * Search delivery patterns by filters with default pagination.
     *
     * @param filters the search criteria.
     * @return the list of matching delivery patterns.
     */
    @PublicApiMember
    List<Delivery> searchPatterns(DeliveryPatternFilters filters);

    // STAGE ENDPOINTS

    /**
     * Adds a stage to a pattern.
     *
     * @param patternId the pattern identifier.
     * @param stage     the stage to be added to the delivery pattern.
     * @return the created stage.
     */
    @POST
    @Path("/{patternId:.*Delivery[^/]*}/stages")
    @PublicApiMember
    Stage createStage(@PathParam("patternId") String patternId, Stage stage);

    @POST
    @Path("/{patternId:.*Delivery[^/]*}/stages/{position:\\d+}")
    @PublicApiMember
    Stage createStage(@PathParam("patternId") String patternId, Stage stage, @PathParam("position") Integer position);

    @POST
    @Path("/{patternId:.*Delivery[^/]*}/createStage")
    @PublicApiMember
    Stage createStage(@PathParam("patternId") String patternId, CreateDeliveryStage form);

    /**
     * Returns the stages in a given delivery pattern.
     *
     * @param patternId the pattern identifier.
     * @return stages in the delivery pattern.
     */
    @GET
    @Path("/{patternId:.*Delivery[^/]*}/stages")
    @PublicApiMember
    List<Stage> getStages(@PathParam("patternId") String patternId);


    /**
     * Updates a stage in a delivery pattern.
     *
     * @param stage new contents of the stage.
     * @return the updated stage.
     */
    @PublicApiMember
    Stage updateStage(Stage stage);

    /**
     * Updates a stage in a delivery pattern.
     *
     * @param stageId the full stage identifier.
     * @param stage   new contents of the stage.
     * @return the updated stage.
     */
    @PUT
    @Path("/{stageId:.*Stage[^/]*}")
    @PublicApiMember
    Stage updateStage(@PathParam("stageId") String stageId, Stage stage);

    /**
     * Updates a stage in a delivery pattern from as-code.
     *
     * @param stageId the full stage identifier.
     * @param stage   new contents of the stage.
     * @return the updated stage.
     */
    @PUT
    @Path("/{stageId:.*Stage[^/]*}/batched")
    Stage updateStageFromBatch(@PathParam("stageId") String stageId, Stage stage);

    /**
     * Deletes a stage in a delivery pattern.
     *
     * @param stageId the full stage identifier.
     */
    @DELETE
    @Path("/{stageId:.*Stage[^/]*}")
    @PublicApiMember
    void deleteStage(@PathParam("stageId") String stageId);

    // TRANSITION ENDPOINTS

    /**
     * Adds a transition to a stage in a delivery pattern.
     *
     * @param stageId    the full stage identifier.
     * @param transition the transition to be added to a stage.
     * @return the created transition.
     */
    @POST
    @Path("/{stageId:.*Stage[^/]*}/transitions")
    @PublicApiMember
    Transition createTransition(@PathParam("stageId") String stageId, Transition transition);

    /**
     * Updates a transition in a delivery pattern.
     *
     * @param transitionId the full transition identifier.
     * @param transition   new contents of the transition.
     * @return the updated transition.
     */
    @PUT
    @Path("/{transitionId:.*Transition[^/]*}")
    @PublicApiMember
    Transition updateTransition(@PathParam("transitionId") String transitionId, Transition transition);

    /**
     * Updates a transition in a delivery pattern.
     *
     * @param transition new contents of the transition.
     * @return the updated transition.
     */
    @PublicApiMember
    Transition updateTransition(Transition transition);

    /**
     * Deletes a transition in a delivery pattern.
     *
     * @param transitionId the full transition identifier.
     */
    @DELETE
    @Path("/{transitionId:.*Transition[^/]*}")
    @PublicApiMember
    void deleteTransition(@PathParam("transitionId") String transitionId);

    // TRACKED ITEM ENDPOINTS

    /**
     * Adds a tracked item to a delivery pattern.
     *
     * @param patternId the pattern identifier.
     * @param item      the item to be added to the delivery pattern.
     * @return the created item.
     */
    @POST
    @Path("/{patternId:.*Delivery[^/]*}/tracked-items")
    @PublicApiMember
    TrackedItem createTrackedItem(@PathParam("patternId") String patternId, TrackedItem item);

    /**
     * Returns the tracked items associated with given delivery pattern.
     *
     * @param patternId the pattern identifier.
     * @return tracked items associated with the delivery pattern.
     */
    @GET
    @Path("/{patternId:.*Delivery[^/]*}/tracked-items")
    @PublicApiMember
    List<TrackedItem> getTrackedItems(@PathParam("patternId") String patternId);

    /**
     * Updates a tracked item.
     *
     * @param itemId the full item identifier.
     * @param item   new contents of the item.
     * @return the updated item.
     */
    @PUT
    @Path("/{itemId:.*TrackedItem[^/]*}")
    @PublicApiMember
    TrackedItem updateTrackedItem(@PathParam("itemId") String itemId, TrackedItem item);

    /**
     * Updates a tracked item.
     *
     * @param item new contents of the item.
     * @return the updated item.
     */
    @PublicApiMember
    TrackedItem updateTrackedItem(TrackedItem item);

    /**
     * Deletes a tracked item.
     *
     * @param itemId the full item identifier.
     */
    @DELETE
    @Path("/{itemId:.*TrackedItem[^/]*}")
    @PublicApiMember
    void deleteTrackedItem(@PathParam("itemId") String itemId);

}
