package com.xebialabs.xlrelease.api.v1;

import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.filter.EnvironmentFilters;
import com.xebialabs.xlrelease.api.v1.form.EnvironmentForm;
import com.xebialabs.xlrelease.api.v1.views.BaseApplicationView;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentReservationView;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentView;
import com.xebialabs.xlrelease.domain.environments.Application;
import com.xebialabs.xlrelease.domain.environments.Environment;
import com.xebialabs.xlrelease.domain.environments.EnvironmentReservation;
import com.xebialabs.xlrelease.rest.AllCILevels;

@Path("/api/v1/environments")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@AllCILevels
@PublicApi
@ShowOnlyPublicApiMembers
public interface EnvironmentApi extends ApiService {

    default String serviceName() {
        return "environmentApi";
    }

    @POST
    EnvironmentView createEnvironment(EnvironmentForm environmentForm);

    @GET
    @Path("/{environmentId:.*/Environment[^/]*}")
    EnvironmentView getEnvironment(@PathParam("environmentId") String environmentId);

    @PUT
    @Path("/{environmentId:.*/Environment[^/]*}")
    EnvironmentView updateEnvironment(@PathParam("environmentId") String environmentId, EnvironmentForm environmentForm);

    @POST
    @Path("/search")
    List<EnvironmentView> searchEnvironments(EnvironmentFilters environmentFilters);

    @GET
    @Path("/{environmentId:.*/Environment[^/]*}/reservations")
    List<EnvironmentReservationView> getReservationsForEnvironment(@PathParam("environmentId") String environmentId);

    @GET
    @Path("/{environmentId:.*/Environment[^/]*}/applications")
    List<BaseApplicationView> getDeployableApplicationsForEnvironment(@PathParam("environmentId") String environmentId);

    /**
     * Deletes an environment.
     *
     * @param environmentId the environment identifier
     */
    @DELETE
    @PublicApiMember
    @Path("/{environmentId:.*/Environment[^/]*}")
    void delete(@PathParam("environmentId") String environmentId);

    /**
     * Creates a new environment.
     *
     * @param environment an <code>Environment</code> object describing the new environment
     * @return created <code>Environment</code> object
     */
    @PublicApiMember
    Environment create(Environment environment);

    /**
     * Gets an environment by id.
     *
     * @param environmentId the environment identifier
     * @return found <code>Environment</code> object
     */
    @PublicApiMember
    Environment getById(String environmentId);

    /**
     * Updates an existing environment.
     *
     * @param environment an <code>Environment</code> object describing the new properties of the environment
     * @return updated <code>Environment</code> object
     */
    @PublicApiMember
    Environment update(Environment environment);

    /**
     * Searches environments by filters.
     *
     * @param filters an <code>EnvironmentFilters</code> object describing the search criteria
     * @return list of matching <code>Environment</code>s
     */
    @PublicApiMember
    List<Environment> search(EnvironmentFilters filters);

    /**
     * Gets all environment reservations for a given environment.
     *
     * @param environmentId the environment identifier
     * @return list of matching <code>EnvironmentReservation</code>s
     */
    @PublicApiMember
    List<EnvironmentReservation> getReservations(String environmentId);

    /**
     * Gets all applications that are allowed to be deployed for a given environment
     *
     * @param environmentId the environment identifier
     * @return list of matching <code>Application</code>s
     */
    @PublicApiMember
    List<Application> getDeployableApplications(String environmentId);
}
