package com.xebialabs.xlrelease.api.v1;

import java.util.Date;
import java.util.List;
import java.util.Map;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;

import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.xlplatform.documentation.PublicApiMember;
import com.xebialabs.xlplatform.documentation.ShowOnlyPublicApiMembers;
import com.xebialabs.xlrelease.api.ApiService;
import com.xebialabs.xlrelease.api.v1.filter.ReservationFilters;
import com.xebialabs.xlrelease.api.v1.form.EnvironmentReservationForm;
import com.xebialabs.xlrelease.api.v1.views.EnvironmentReservationView;
import com.xebialabs.xlrelease.api.v1.views.search.EnvironmentReservationSearchView;
import com.xebialabs.xlrelease.domain.environments.EnvironmentReservation;
import com.xebialabs.xlrelease.rest.AllCILevels;

@Path("/api/v1/environments/reservations")
@AllCILevels
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@PublicApi
@ShowOnlyPublicApiMembers
public interface EnvironmentReservationApi extends ApiService {

    default String serviceName() {
        return "environmentReservationApi";
    }

    @POST
    EnvironmentReservationView createReservation(EnvironmentReservationForm environmentReservationForm);

    @GET
    @Path("/{environmentReservationId:.*/EnvironmentReservation[^/]*}")
    EnvironmentReservationView getReservation(@PathParam("environmentReservationId") String environmentReservationId);

    @PUT
    @Path("/{environmentReservationId:.*/EnvironmentReservation[^/]*}")
    EnvironmentReservationView updateReservation(@PathParam("environmentReservationId") String environmentReservationId,
                                                 EnvironmentReservationForm environmentReservationForm);

    @POST
    @Path("/search")
    List<EnvironmentReservationSearchView> searchReservations(ReservationFilters filters);

    /**
     * Deletes an environment reservation.
     *
     * @param environmentReservationId the environment reservation identifier
     */
    @DELETE
    @PublicApiMember
    @Path("/{environmentReservationId:.*/EnvironmentReservation[^/]*}")
    void delete(@PathParam("environmentReservationId") String environmentReservationId);

    /**
     * Adds an application to an environment reservation.
     *
     * @param environmentReservationId the environment reservation identifier
     * @param applicationId            the application identifier
     */
    @POST
    @PublicApiMember
    @Path("/{environmentReservationId:.*/EnvironmentReservation[^/]*}")
    void addApplication(@PathParam("environmentReservationId") String environmentReservationId, @QueryParam("applicationId") String applicationId);

    /**
     * Creates a new environment reservation.
     *
     * @param environmentReservation an <code>EnvironmentReservation</code> object describing the new environment reservation
     * @return created <code>EnvironmentReservation</code> object
     */
    @PublicApiMember
    EnvironmentReservation create(EnvironmentReservation environmentReservation);

    /**
     * Gets an environment reservation by id.
     *
     * @param environmentReservationId the environment reservation identifier
     * @return found <code>EnvironmentReservation</code> object
     */
    @PublicApiMember
    EnvironmentReservation getById(String environmentReservationId);

    /**
     * Updates an existing environment reservation.
     *
     * @param environmentReservation an <code>EnvironmentReservation</code> object describing the new properties of the environment reservation
     * @return updated <code>EnvironmentReservation</code> object
     */
    @PublicApiMember
    EnvironmentReservation update(EnvironmentReservation environmentReservation);

    /**
     * Searches environment reservations by filters.
     *
     * @param filters an <code>EnvironmentReservationFilters</code> object describing the search criteria
     * @return map of matching <code>EnvironmentReservation</code> lists grouped by environment IDs
     */
    @PublicApiMember
    Map<String, List<EnvironmentReservation>> search(ReservationFilters filters);

    /**
     * Checks if given application has a reservation on a given environment at the current time.
     *
     * @param environmentId the environment identifier
     * @param applicationId the application identifier
     * @return true if a reservation exists
     */
    @PublicApiMember
    boolean hasReservation(String environmentId, String applicationId);

    /**
     * Checks if given application has a reservation on a given environment at the current time or
     * in the future.
     *
     * @param environmentId the environment identifier
     * @param applicationId the application identifier
     * @return Date of reservation start time.
     * If reservation exists at the current time, then returned date will be less than or equal to now.
     * If future reservation is found, then returned date will be start date of nearest future reservation
     * If neither current nor future reservation found, return value is null
     */
    @PublicApiMember
    Date nearestComingReservation(String environmentId, String applicationId);
}
