package com.xebialabs.xlrelease.support.cache.caffeine

import com.github.benmanes.caffeine.cache.stats.CacheStats
import com.github.benmanes.caffeine.cache.{Cache, Policy}
import grizzled.slf4j.Logging

import java.util.concurrent.ConcurrentMap
import java.util.{Collections, function}
import java.{lang, util}
import scala.jdk.CollectionConverters._

class CaffeineCacheDelegate[K, V](name: String, cache: Cache[K, V],
                                  invalidateWithRegex: Boolean,
                                  var enabled: Boolean) extends Cache[K, V] with Logging {

  def enable(): Unit = {
    logger.trace(s"Enabling cache [$name]")
    enabled = true
  }

  def disable(): Unit = {
    logger.trace(s"Disabling cache [$name] and clearing it")
    enabled = false
    this.invalidateAll()
  }

  override def getIfPresent(key: K): V = {
    if (enabled) {
      cache.getIfPresent(key)
    } else {
      null.asInstanceOf[V]
    }
  }

  override def get(key: K, mappingFunction: function.Function[_ >: K, _ <: V]): V = {
    if (enabled) {
      cache.get(key, mappingFunction)
    } else {
      mappingFunction(key)
    }
  }

  override def getAllPresent(keys: lang.Iterable[_ <: K]): util.Map[K, V] = {
    if (enabled) {
      cache.getAllPresent(keys)
    } else {
      Collections.emptyMap()
    }
  }

  override def put(key: K, value: V): Unit = if (enabled) {
    cache.put(key, value)
  }

  override def putAll(map: util.Map[_ <: K, _ <: V]): Unit = if (enabled) {
    cache.putAll(map)
  }

  override def invalidate(key: K): Unit = {
    if (invalidateWithRegex) {
      this.invalidateAll(util.Arrays.asList(key))
    } else {
      cache.invalidate(key)
    }
  }

  override def invalidateAll(keys: lang.Iterable[_ <: K]): Unit = {
    if (invalidateWithRegex) {
      logger.trace(s"Invalidating $keys from cache [$name]")
      val existingKeys = cache.asMap().keySet().asScala.filter(_ != null).map(_.toString)
      val existingKeysToRemove = keys.asScala.filter(_ != null).flatMap(keyToRemove => {
        existingKeys.filter(existingKey => existingKey == keyToRemove || (keyToRemove.toString.startsWith("regex:") && existingKey.matches(keyToRemove.toString.replace("regex:", ""))))
      }).asJava
      cache.invalidateAll(existingKeysToRemove.asInstanceOf[lang.Iterable[_ <: K]])
      logger.trace(s"Invalidated $existingKeysToRemove from cache [$name]")
    } else {
      cache.invalidateAll(keys)
    }
  }

  override def invalidateAll(): Unit = {
    logger.trace(s"Invalidating all from cache [$name]")
    cache.invalidateAll()
  }

  override def estimatedSize(): Long = cache.estimatedSize()

  override def stats(): CacheStats = cache.stats()

  override def asMap(): ConcurrentMap[K, V] = cache.asMap()

  override def cleanUp(): Unit = cache.cleanUp()

  override def policy(): Policy[K, V] = cache.policy()

  override def getAll(keys: lang.Iterable[_ <: K], mappingFunction: function.Function[_ >: util.Set[_ <: K], _ <: util.Map[_ <: K, _ <: V]]): util.Map[K, V] = {
    if (enabled) {
      cache.getAll(keys, mappingFunction)
    } else {
      Collections.emptyMap()
    }
  }
}
