package com.xebialabs.xlrelease.script;

import java.time.Duration;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

public final class ScriptTimeoutContext {
    private static final String DOCUMENTATION_REF_MESSAGE = "Refer to Digital.ai Release documentation to increase timeouts.";
    private static final String DEFAULT_TIMEOUT_MESSAGE = "Task script timed out.";

    private final boolean timeoutEnabled;
    private final Duration timeout;
    private final ScheduledExecutorService timeoutExecutor;
    private final AtomicBoolean completionStatus = new AtomicBoolean(false);
    private volatile boolean timeoutHappened;
    private final String timeoutMessage;

    public ScriptTimeoutContext(boolean timeoutEnabled, ScheduledExecutorService timeoutExecutor, Duration timeout) {
        this.timeoutEnabled = timeoutEnabled;
        this.timeoutExecutor = timeoutExecutor;
        this.timeout = timeout;
        this.timeoutMessage = DEFAULT_TIMEOUT_MESSAGE;
    }

    public ScriptTimeoutContext(boolean timeoutEnabled, ScheduledExecutorService timeoutExecutor, Duration timeout, String timeoutMessage) {
        this.timeoutEnabled = timeoutEnabled;
        this.timeoutExecutor = timeoutExecutor;
        this.timeout = timeout;
        this.timeoutMessage = timeoutMessage;
    }

    void registerTimeout() {
        Thread executionThread = Thread.currentThread();
        if (this.timeoutEnabled && this.timeoutExecutor != null) {
            this.timeoutExecutor.schedule(() -> interruptScript(executionThread), this.timeout.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    private synchronized void interruptScript(Thread executionThread) {
        if (this.completionStatus.compareAndSet(false, true)) {
            this.timeoutHappened = true;
            executionThread.interrupt();
        }
    }

    synchronized void cleanupTimeout() {
        if (this.timeoutEnabled && !this.completionStatus.compareAndSet(false, true)) {
            // if previously completion status was not false it means that timeout already happened
            // and thread was already interrupted (we can be sure about it because methods are synchronized)
            Thread.interrupted();
        }
    }

    boolean isTimedOut() {
        return this.timeoutHappened;
    }

    String getTimeoutMessage() {
        return String.format("%s %s", timeoutMessage, DOCUMENTATION_REF_MESSAGE);
    }

    public static ScriptTimeoutContext noTimeoutContext() {
        return new ScriptTimeoutContext(false, null, Duration.ZERO);
    }
}
