package com.xebialabs.xlrelease.security.sql

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionEditor, Permissions, Role}
import com.xebialabs.xlplatform.repository.sql.Database
import com.xebialabs.xlrelease.events.Subscribe
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationConstants._
import com.xebialabs.xlrelease.security.sql.db.Ids._
import com.xebialabs.xlrelease.security.sql.db.Tables
import com.xebialabs.xlrelease.security.sql.db.Tables.RolePermission
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable, Caching}
import slick.dbio.DBIOAction.seq
import slick.jdbc.JdbcProfile
import slick.lifted.Query

import java.{lang, util}
import scala.jdk.CollectionConverters._

@CacheConfig(cacheManager = SECURITY_CACHE_MANAGER)
class CachingPermissionEditor(securityDatabase: Database, broadcastService: BroadcastService)
  extends SqlPermissionEditor(securityDatabase) {

  @Subscribe
  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = "'regex:' + (#onConfigurationItem?:'global') + '-.*'")
  ))
  override def editPermissions(onConfigurationItem: String, permissions: util.Map[Role, util.Set[Permission]]): Unit = {
    super.editPermissions(onConfigurationItem, permissions)
    expireCache(onConfigurationItem)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = "'regex:' + (#onConfigurationItem?:'global') + '-.*'")
  ))
  override def updatePermissions(onConfigurationItem: String,
                                 addedPermissions: util.Map[Role, util.List[Permission]],
                                 removedPermissions: util.Map[Role, util.List[Permission]]): Unit = {
    super.updatePermissions(onConfigurationItem, addedPermissions, removedPermissions)
    expireCache(onConfigurationItem)
  }

  @Cacheable(cacheNames = Array(SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL)
  override def readPermissions(onConfigurationItem: String, includeInherited: lang.Boolean): util.Map[Role, util.Set[Permission]] = {
    super.readPermissions(onConfigurationItem, includeInherited)
  }

  @Cacheable(cacheNames = Array(SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL)
  override def readPermissions(onConfigurationItem: String): util.Map[Role, util.Set[Permission]] = {
    super.readPermissions(onConfigurationItem)
  }

  private def expireCache(onConfigurationItem: String): Unit = {
    val target = if (isGlobalId(Option(onConfigurationItem))) GLOBAL_SECURITY_ALIAS else onConfigurationItem
    broadcastService.broadcast(EvictPermissionsEvent(target), false)
  }
}

class SqlPermissionEditor(securityDatabase: Database) extends PermissionEditor with Logging {

  import securityDatabase._

  type Q = Query[(Tables.Roles, Tables.RolePermissions), (Tables.Role, RolePermission), Seq]

  val profile: JdbcProfile = config.databaseType.profile

  import profile.api._

  override def editPermissions(onConfigurationItem: String, permissions: util.Map[Role, util.Set[Permission]]): Unit = {
    logger.debug(s"Writing permissions $permissions to CI [$onConfigurationItem]")

    val notApplicableTo = Permissions.isApplicableTo(permissions.asScala.flatMap(_._2.asScala).toList.asJava, onConfigurationItem, false)
    checkArgument(notApplicableTo.isEmpty, "The permissions %s are not applicable to [%s]", notApplicableTo, onConfigurationItem)

    runAwait {
      seq(
        Tables.rolePermissions.filter(_.isOnConfigurationItem(Option(onConfigurationItem))).delete,
        Tables.rolePermissions ++= permissions.asScala.flatMap {
          case (role, rolePermissions) =>
            rolePermissions.asScala.map(permission => RolePermission(role.getId, permission.getPermissionName, toDbId(Option(onConfigurationItem))))
        }
      ).transactionally
    }
  }

  private def withRolePattern(query: Q, rolePattern: String) = Option(rolePattern) match {
    case Some(pattern) if pattern.nonEmpty => query.filter { case (role, _) => role.name.toLowerCase.like(s"%${rolePattern.toLowerCase}%") }
    case _ => query
  }

  private def withPaging(query: Q, paging: Paging) = Option(paging) match {
    case Some(p) => query
      .drop((p.page - 1) * p.resultsPerPage)
      .take(p.resultsPerPage)
    case _ => query
  }

  private def withOrder(query: Q, order: Ordering) = Option(order) match {
    case Some(ord) => if (ord.isAscending) query.sortBy(_._1.name.toLowerCase.asc) else query.sortBy(_._1.name.toLowerCase.desc)
    case _ => query
  }

  override def readPermissions(onConfigurationItem: String,
                               rolePattern: String,
                               paging: Paging,
                               order: Ordering,
                               includeInherited: lang.Boolean): util.Map[Role, util.Set[Permission]] = {
    logger.debug(s"Reading permissions from [$onConfigurationItem]")

    var query: Q = Tables.roles
      .join(Tables.rolePermissions).on(_.id === _.roleId)
      .filter { case (_, rolePermission) => rolePermission.isOnConfigurationItem(Option(onConfigurationItem)) }

    query = withRolePattern(query, rolePattern)
    query = withOrder(query, order)
    query = withPaging(query, paging)

    val permissions = runAwait {
      query.result
    }
      .map { case (role, rolePermission) => new Role(role.id, role.name) -> Option(Permission.find(rolePermission.permissionName)) }
      .collect { case (role, Some(permission)) => role -> permission }
      .groupBy(_._1)
      .view
      .mapValues { values => values.map(_._2).toSet.asJavaMutable() }
      .toMap

    logger.debug(s"Read from [$onConfigurationItem] permissions $permissions")
    permissions.asJavaMutable()
  }

  override def updatePermissions(onConfigurationItem: String,
                                 addedPermissions: util.Map[Role, util.List[Permission]],
                                 removedPermissions: util.Map[Role, util.List[Permission]]): Unit = {
    logger.debug(s"Adding permissions $addedPermissions and removing permissions $removedPermissions to CI [$onConfigurationItem]")

    val notApplicableTo = Permissions.isApplicableTo(addedPermissions.asScala.flatMap(_._2.asScala).toList.asJava, onConfigurationItem, false)
    checkArgument(notApplicableTo.isEmpty, "The permissions %s are not applicable to [%s]", notApplicableTo, onConfigurationItem)

    removedPermissions.forEach { case (role, permissions) =>
      val permissionNames = permissions.asScala.map(_.getPermissionName)
      runAwait {
        seq(
          Tables.rolePermissions.filter(_.roleId in List(role.getId)).filter(_.permissionName in permissionNames).delete,
        ).transactionally
      }
    }

    runAwait {
      seq(
        Tables.rolePermissions ++= addedPermissions.asScala.flatMap {
          case (role, rolePermissions) =>
            rolePermissions.asScala.map(permission => RolePermission(role.getId, permission.getPermissionName, toDbId(Option(onConfigurationItem))))
        }
      ).transactionally
    }
  }
}
