package com.xebialabs.xlrelease.security.sql

import com.xebialabs.deployit.security.Permissions.getAuthentication
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PlatformPermissions.LOGIN
import com.xebialabs.deployit.security.{PermissionChecker, Role}
import com.xebialabs.xlplatform.repository.sql.Database
import com.xebialabs.xlrelease.security.sql.db.Ids.isGlobalId
import com.xebialabs.xlrelease.security.sql.db.Tables
import grizzled.slf4j.Logging
import org.springframework.security.core.Authentication
import slick.jdbc.JdbcProfile

import java.{lang, util}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

class SqlPermissionChecker(securityDatabase: Database, permissionLister: SqlPermissionLister, isCacheEnabled: Boolean = false) extends PermissionChecker with Logging {

  import securityDatabase._

  val profile: JdbcProfile = config.databaseType.profile

  import profile.api._

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, roles: util.List[Role]): Boolean =
    checkPermission(permissions, onConfigurationItem, roles, getAuthentication)

  override def checkPermission(permissions: util.List[Permission], onConfigurationItem: String, roles: util.List[Role], auth: Authentication): Boolean =
    checkPermission(permissions.asScala.toSeq, Option(onConfigurationItem), roles.asScala.toSeq, auth)

  override def checkPermission(permissions: util.List[Permission], onConfigurationItems: util.List[String], allRoles: util.List[Role]): util.Map[String, lang.Boolean] =
    checkPermission(permissions, onConfigurationItems, allRoles, getAuthentication)

  override def checkPermission(permissions: util.List[Permission], onConfigurationItems: util.List[String], allRoles: util.List[Role], auth: Authentication): util.Map[String, lang.Boolean] =
    onConfigurationItems.asScala.toList.foldLeft(mutable.Map.empty[String, Boolean]) { (acc, ci) =>
      val hasPermission = checkPermission(permissions, ci, allRoles, auth)
      acc.put(ci, hasPermission)
      acc
    }.view.mapValues(Boolean.box).toMap.asJava

  protected def checkPermission(permissions: Seq[Permission], onConfigurationItem: Option[String], roles: Seq[Role], auth: Authentication): Boolean = {
    val target = if (isGlobalId(onConfigurationItem)) "global" else onConfigurationItem.get
    logger.debug(s"Checking permissions [${permissions.mkString(", ")}] on [$target]")

    var hasPermission = hasPermissions(permissions, onConfigurationItem, roles, auth, target)

    // backwards-compatibility: login permission is assigned to everyone if no other roles are assigned
    if (!hasPermission && isLoginAssignedToEveryone(permissions, onConfigurationItem)) {
      logger.debug(s"Login permission is granted to everyone")
      hasPermission = true
    }
    logger.debug(s"Permissions [${permissions.mkString(", ")}] granted on [$target]: $hasPermission")

    hasPermission
  }

  private def hasPermissions(permissions: Seq[Permission], onConfigurationItem: Option[String], roles: Seq[Role], auth: Authentication, target: String) = {
    if (isCacheEnabled) {
      val assignedPermissions = permissionLister.listUserPermissionsFor(target, auth, roles)
      permissions.exists(permission => assignedPermissions.contains(permission.getPermissionName))
    } else {
      val allRoles = if (isGlobalId(onConfigurationItem)) roles else roles ++ permissionLister.getLocalRolesFor(onConfigurationItem, auth, roles)

      runAwait {
        Tables.roles
          .join(Tables.rolePermissions).on(_.id === _.roleId)
          .filter { case (role, rolePermissions) =>
            role.id.inLarge(allRoles.map(_.getId)) &&
              rolePermissions.isOnConfigurationItem(onConfigurationItem) &&
              rolePermissions.permissionName.in(permissions.map(_.getPermissionName))
          }.exists.result
      }
    }
  }

  protected def isLoginAssignedToEveryone(permissions: Seq[Permission], onConfigurationItem: Option[String]): Boolean =
    permissions.nonEmpty && permissions.forall(_.getPermissionName == LOGIN.getPermissionName) && !checkAnyRolesAssignedToPermission(onConfigurationItem, LOGIN)

  protected def checkAnyRolesAssignedToPermission(onConfigurationItem: Option[String], permission: Permission): Boolean = {
    runAwait {
      Tables.rolePermissions
        .filter(rolePermission => rolePermission.isOnConfigurationItem(onConfigurationItem) && rolePermission.permissionName === permission.getPermissionName)
        .exists
        .result
    }
  }
}
