package com.xebialabs.xlrelease.security.sql

import com.xebialabs.deployit.security.Role
import com.xebialabs.xlplatform.repository.sql.Database
import com.xebialabs.xlrelease.config.CacheManagementConstants.SECURITY_CACHE_MANAGER
import com.xebialabs.xlrelease.domain.distributed.events.{EvictAllEvent, EvictRolesEvent}
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationConstants._
import com.xebialabs.xlrelease.security.sql.db.Ids.{GLOBAL_SECURITY_ALIAS, isGlobalId}
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.cache.annotation.{CacheConfig, CacheEvict, Cacheable, Caching}
import org.springframework.security.core.Authentication

import java.util

@CacheConfig(cacheManager = SECURITY_CACHE_MANAGER)
class CachingSqlRoleService(securityDatabase: Database, broadcastService: BroadcastService) extends SqlRoleService(securityDatabase) with Logging {

  @Cacheable(cacheNames = Array(SECURITY_ROLES), key = GLOBAL_KEY)
  override def getRoles(): util.List[Role] = super.getRoles()

  @Cacheable(cacheNames = Array(SECURITY_ROLES))
  override def getRoles(onConfigurationItem: String): util.List[Role] = super.getRoles(onConfigurationItem)

  @Cacheable(cacheNames = Array(SECURITY_USER_ROLES), key = "'global-' + #auth?.name")
  override def getRolesFor(auth: Authentication): util.List[Role] = super.getRolesFor(auth)

  @Cacheable(cacheNames = Array(SECURITY_USER_ROLES), key = "'global-' + #principal")
  override def getRolesFor(principal: String): util.List[Role] = super.getRolesFor(principal)

  @Cacheable(cacheNames = Array(SECURITY_PERMISSION_ROLES), key = "'global-' + #permission")
  override def getRolesForPermission(permission: String): util.List[Role] = super.getRolesForPermission(permission)

  @Cacheable(cacheNames = Array(SECURITY_ROLE_ASSIGNMENTS), key = GLOBAL_KEY)
  override def readRoleAssignments(): util.List[Role] = super.readRoleAssignments()

  @Cacheable(cacheNames = Array(SECURITY_ROLE_ASSIGNMENTS))
  override def readRoleAssignments(onConfigurationItem: String): util.List[Role] = super.readRoleAssignments(onConfigurationItem)

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true),
  ))
  override def writeRoleAssignments(roles: util.List[Role]): Unit = {
    super.writeRoleAssignments(roles)
    expireCache(GLOBAL_SECURITY_ALIAS)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def writeRoleAssignments(onConfigurationItem: String, roles: util.List[Role]): Unit = {
    super.writeRoleAssignments(onConfigurationItem, roles)
    expireCache(onConfigurationItem)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def createOrUpdateRole(role: Role, onConfigurationItem: String): String = {
    val roleId = super.createOrUpdateRole(role, onConfigurationItem)
    expireCache(onConfigurationItem)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def rename(name: String, newName: String, onConfigurationItem: String): String = {
    val roleId = super.rename(name, newName, onConfigurationItem)
    expireCache(onConfigurationItem)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def create(roles: Role*): Unit = {
    super.create(roles: _*)
    expireCache(GLOBAL_SECURITY_ALIAS)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def create(onConfigurationItem: String, roles: Role*): Unit = {
    super.create(onConfigurationItem, roles: _*)
    expireCache(onConfigurationItem)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def create(name: String, onConfigurationItem: String): String = {
    val roleId = super.create(name, onConfigurationItem)
    expireCache(onConfigurationItem)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def create(name: String): String = {
    val roleId: String = super.create(name)
    expireCache(GLOBAL_SECURITY_ALIAS)
    roleId
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def update(roles: Role*): Unit = {
    super.update(roles: _*)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def update(onConfigurationItem: String, roles: Role*): Unit = {
    super.update(roles: _*)
    expireCache(onConfigurationItem)
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def deleteByName(name: String): Unit = {
    super.deleteByName(name)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def deleteById(roleId: String): Unit = {
    super.deleteById(roleId)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @CacheEvict(cacheNames = Array(
    SECURITY_ROLES,
    SECURITY_ROLE_ASSIGNMENTS,
    SECURITY_PERMISSIONS,
    SECURITY_USER_ROLES,
    SECURITY_USER_PERMISSIONS,
    SECURITY_PERMISSION_ROLES
  ), allEntries = true)
  override def delete(roleIds: String*): Unit = {
    super.delete(roleIds: _*)
    expireCache(GLOBAL_SECURITY_ALIAS, allEntries = true)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = ON_CI_OR_GLOBAL),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), key = CACHE_EVICTION_KEY_GENERATION)
  ))
  override def deleteRoles(onConfigurationItem: String, roleIds: String*): Unit = {
    super.delete(roleIds: _*)
    expireCache(onConfigurationItem)
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def createOrUpdateRole(role: Role): String = {
    val roleId = super.createOrUpdateRole(role)
    expireCache(GLOBAL_SECURITY_ALIAS)
    roleId
  }

  @Caching(evict = Array(
    new CacheEvict(cacheNames = Array(SECURITY_ROLES, SECURITY_ROLE_ASSIGNMENTS, SECURITY_PERMISSIONS), key = GLOBAL_KEY),
    new CacheEvict(cacheNames = Array(SECURITY_USER_ROLES, SECURITY_USER_PERMISSIONS, SECURITY_PERMISSION_ROLES), allEntries = true)
  ))
  override def rename(name: String, newName: String): String = {
    val roleId = super.rename(name, newName)
    expireCache(GLOBAL_SECURITY_ALIAS)
    roleId
  }

  private def expireCache(onConfigurationItem: String, allEntries: Boolean = false): Unit = {
    val target = if (isGlobalId(Option(onConfigurationItem))) GLOBAL_SECURITY_ALIAS else onConfigurationItem
    broadcastService.broadcast(if (allEntries) EvictAllEvent() else EvictRolesEvent(target), publishEventOnSelf = false)
  }
}
