package com.xebialabs.xlrelease.repository.sql.persistence.configuration

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlplatform.utils.ResourceManagement
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.utils.IdUtils
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{RichBooleanAsInt, params}
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, PersistenceSupport}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.Try

object ConfigurationPersistence {
  type ConfigurationRow = (Option[FolderId], String)

  val mapFolderPath: ResultSet => Option[FolderId] = (rs: ResultSet) => {
    for {
      path <- Option(rs.getString(FOLDERS.FOLDER_PATH))
      id <- Option(rs.getString(FOLDERS.FOLDER_ID))
    } yield FolderId(path) / id
  }
}

class ConfigurationPersistence(override val folderPersistence: FolderPersistence,
                               override val jdbcTemplate: JdbcTemplate,
                               override implicit val dialect: Dialect)
  extends ConfigurationPersistenceCommon

private[configuration] abstract class ConfigurationPersistenceCommon extends SqlRepository with PersistenceSupport with LimitOffset with Logging {
  def folderPersistence: FolderPersistence

  override implicit val dialect: Dialect

  private val STMT_GET_UID_BY_ID = s"SELECT ${CONFIGURATIONS.CI_UID} FROM ${CONFIGURATIONS.TABLE} WHERE ${CONFIGURATIONS.ID} = :configurationId"

  def getUid(configurationId: String): Option[CiUid] =
    sqlQuery(STMT_GET_UID_BY_ID, params("configurationId" -> configurationId), rs => CiUid(rs.getInt(CONFIGURATIONS.CI_UID))).headOption

  private val STMT_GET_TYPE_BY_ID = s"SELECT ${CONFIGURATIONS.CI_TYPE} FROM ${CONFIGURATIONS.TABLE} WHERE ${CONFIGURATIONS.ID} = :configurationId"

  def getType(configurationId: String): Option[Type] = {
    sqlQuery(STMT_GET_TYPE_BY_ID, params("configurationId" -> configurationId), rs => Type.valueOf(rs.getString(CONFIGURATIONS.CI_TYPE))).headOption
  }

  private val STMT_INSERT_CONFIGURATION =
    s"""|INSERT INTO ${CONFIGURATIONS.TABLE}
        |   ( ${CONFIGURATIONS.ID}
        |   , ${CONFIGURATIONS.TITLE}
        |   , ${CONFIGURATIONS.CI_TYPE}
        |   , ${CONFIGURATIONS.FOLDER_CI_UID}
        |   , ${CONFIGURATIONS.CONTENT}
        |   )
        | VALUES
        |   ( :configurationId
        |   , :title
        |   , :ciType
        |   , :folderCiUid
        |   , :content
        |   )
        """.stripMargin

  private def getFolderCiUid(folderId: String): CiUid = {
    folderId match {
      case null => null
      case id => folderPersistence.findById(id, depth = 0).value.uid
    }
  }

  def insert(configuration: BaseConfiguration): Unit = {
    val parentId: String = configuration.getFolderId
    insert(configuration, getFolderCiUid(parentId))
  }

  def insert(configuration: BaseConfiguration, folderCiUid: CiUid): Unit = {
    val parentId: String = configuration.getFolderId

    configuration.setFolderId(null)

    IdUtils.createMissingNestedIds(configuration)

    val content = serialize(configuration)
    try {
      val parmap = params(
        "configurationId" -> configuration.getId,
        "title" -> configuration.getTitle,
        "ciType" -> configuration.getType.toString,
        "folderCiUid" -> folderCiUid
      )
      sqlExecWithContent(STMT_INSERT_CONFIGURATION, parmap, "content" -> content,
        _ => configuration.setFolderId(parentId))
    } catch {
      case ex: DuplicateKeyException => throw new IllegalArgumentException(s"Configuration item with id '${configuration.getId}' already exists", ex)
    }
  }

  private val STMT_UPDATE_CONFIGURATION =
    s"""|UPDATE ${CONFIGURATIONS.TABLE}
        | SET
        |  ${CONFIGURATIONS.TITLE} = :newTitle,
        |  ${CONFIGURATIONS.CI_TYPE} = :ciType,
        |  ${CONFIGURATIONS.CONTENT} = :content,
        |  ${CONFIGURATIONS.FOLDER_CI_UID} = :folderCiUid
        | WHERE
        |  ${CONFIGURATIONS.ID} = :configurationId
        """.stripMargin

  def update(configuration: BaseConfiguration): Unit = {
    update(configuration, getFolderCiUid(configuration.getFolderId))
  }

  def update(configuration: BaseConfiguration, folderCiUid: CiUid): Unit = {
    val previousFolderId = configuration.getFolderId
    configuration.setFolderId(null)

    val content = serialize(configuration)
    sqlExecWithContent(STMT_UPDATE_CONFIGURATION, params(
      "configurationId" -> configuration.getId,
      "newTitle" -> configuration.getTitle,
      "ciType" -> configuration.getType.toString,
      "folderCiUid" -> folderCiUid
    ), "content" -> content,
      {
        configuration.setFolderId(previousFolderId)
        checkCiUpdated(configuration.getId)
      }
    )
  }

  private val STMT_DELETE_BY_ID =
    s"""|DELETE FROM ${CONFIGURATIONS.TABLE}
        | WHERE ${CONFIGURATIONS.ID} = :configurationId""".stripMargin

  def delete(configurationId: String): Try[Boolean] =
    sqlExec(STMT_DELETE_BY_ID, params("configurationId" -> configurationId), ps => Try(ps.execute()))

  private val STMT_DELETE_BY_UID =
    s"""|DELETE FROM ${CONFIGURATIONS.TABLE}
        | WHERE ${CONFIGURATIONS.CI_UID} = :configurationUid""".stripMargin

  def deleteByUid(configurationUid: CiUid): Boolean = {
    sqlExec(STMT_DELETE_BY_UID, params("configurationUid" -> configurationUid), _.execute())
  }

  private val STMT_EXISTS: String = s"SELECT COUNT(*) FROM ${CONFIGURATIONS.TABLE} WHERE ${CONFIGURATIONS.ID} = :configurationId"

  def exists(configurationId: String): Boolean = {
    sqlQuery(STMT_EXISTS, params("configurationId" -> configurationId), _.getInt(1) > 0).head
  }

  private val STMT_EXISTS_BY_TYPE_TITLE: String =
    s"""|SELECT
        |   COUNT(*)
        | FROM ${CONFIGURATIONS.TABLE} conf
        | WHERE LOWER(conf.${CONFIGURATIONS.TITLE}) = :title AND conf.${CONFIGURATIONS.CI_TYPE} IN (:types)""".stripMargin

  def existsByTypeAndTitle(ciTypes: Seq[String], title: String): Boolean =
    sqlQuery(STMT_EXISTS_BY_TYPE_TITLE, params(
      "title" -> title.toLowerCase(),
      "types" -> ciTypes.asJava
    ), _.getInt(1) > 0).head

  private val STMT_FIND_BY_ID =
    s"""|SELECT
        |   conf.${CONFIGURATIONS.CONTENT},
        |   f.${FOLDERS.FOLDER_PATH},
        |   f.${FOLDERS.FOLDER_ID}
        | FROM ${CONFIGURATIONS.TABLE} conf
        | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = conf.${CONFIGURATIONS.FOLDER_CI_UID}
        | WHERE conf.${CONFIGURATIONS.ID} = :configurationId""".stripMargin

  def findById(configurationId: String): Option[ConfigurationRow] = {
    logger.trace(s"Loading configuration with id: $configurationId")
    sqlQuery(STMT_FIND_BY_ID, params("configurationId" -> configurationId), configurationBinaryStreamRowMapper).headOption
  }

  val STMT_FIND_BY_TYPE_TITLE_IN_APPLICATIONS: String =
    s"""|SELECT
        |   conf.${CONFIGURATIONS.CONTENT}
        |FROM ${CONFIGURATIONS.TABLE} conf
        |WHERE
        |   (conf.${CONFIGURATIONS.FOLDER_CI_UID} IS NULL) AND
        |   (conf.${CONFIGURATIONS.CI_TYPE} IN (:types)) AND
        |   (:byTitle = 0 OR conf.${CONFIGURATIONS.TITLE} = :title)
        |ORDER BY conf.${CONFIGURATIONS.TITLE} ASC""".stripMargin

  val STMT_FIND_BY_TYPE_TITLE_AND_EXACT_FOLDER: String =
    s"""|SELECT
        |   conf.${CONFIGURATIONS.CONTENT},
        |   f.${FOLDERS.FOLDER_PATH},
        |   f.${FOLDERS.FOLDER_ID}
        |FROM ${CONFIGURATIONS.TABLE} conf
        |LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = conf.${CONFIGURATIONS.FOLDER_CI_UID}
        |WHERE
        |   (conf.${CONFIGURATIONS.CI_TYPE} IN (:types)) AND
        |   (:byTitle = 0 OR conf.${CONFIGURATIONS.TITLE} = :title) AND
        |   (f.${FOLDERS.FOLDER_ID} = :folderId)
        |ORDER BY conf.${CONFIGURATIONS.TITLE} ASC""".stripMargin

  val STMT_FIND_BY_TYPE_TITLE_AND_PARENT_FOLDERS: String =
    s"""|SELECT
        |   conf.${CONFIGURATIONS.CONTENT},
        |   f.${FOLDERS.FOLDER_PATH},
        |   f.${FOLDERS.FOLDER_ID}
        |FROM ${CONFIGURATIONS.TABLE} conf
        |LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = conf.${CONFIGURATIONS.FOLDER_CI_UID}
        |LEFT JOIN ${PATHS.TABLE} p ON p.${PATHS.ANCESTOR_UID} = f.${FOLDERS.CI_UID}
        |LEFT JOIN ${FOLDERS.TABLE} a ON a.${FOLDERS.CI_UID} = p.${PATHS.DESCENDANT_UID}
        |WHERE
        |   (conf.${CONFIGURATIONS.CI_TYPE} IN (:types)) AND
        |   (:byTitle = 0 OR conf.${CONFIGURATIONS.TITLE} = :title) AND
        |   (a.${FOLDERS.FOLDER_ID} = :folderId OR conf.${CONFIGURATIONS.FOLDER_CI_UID} IS NULL)
        |ORDER BY conf.${CONFIGURATIONS.TITLE} ASC""".stripMargin

  /**
   *
   * @param ciTypes  list of CI types to include in the search
   * @param title    None: don't filter by title
   *                 Some(str): only match configuration whose title matches
   * @param folderId None: search only in global scope
   *                 Some(Left(id)): search in all parent folders of 'id', including folder 'id' and global scope
   *                 Some(Right(id)): search only in folder 'id'
   * @return list of ConfigurationRow found, sorted by title. Can be deserialized by SqlConfigurationRepository.
   */
  def findByTypesTitleAndFolder(ciTypes: Seq[String], title: Option[String], folderId: Option[Either[String, String]]): Seq[ConfigurationRow] = {
    val ((query, id), mapper) = folderId.map {
      _.fold(
        STMT_FIND_BY_TYPE_TITLE_AND_PARENT_FOLDERS -> Some(_),
        STMT_FIND_BY_TYPE_TITLE_AND_EXACT_FOLDER -> Some(_)
      ) -> configurationBinaryStreamRowMapper
    }.getOrElse {
      STMT_FIND_BY_TYPE_TITLE_IN_APPLICATIONS -> None -> globalConfigurationBinaryStreamRowMapper
    }
    sqlQuery(query, params(
      "types" -> ciTypes.asJava,
      "byTitle" -> title.nonEmpty.asInteger,
      "title" -> title.orNull,
      "folderId" -> id.map(Ids.getName).orNull
    ), mapper).toSeq
  }

  private val STMT_FIND_BY_TYPE: String =
    s"""|SELECT
        |  conf.${CONFIGURATIONS.CONTENT},
        |  f.${FOLDERS.FOLDER_PATH},
        |  f.${FOLDERS.FOLDER_ID}
        |FROM ${CONFIGURATIONS.TABLE} conf
        |LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = conf.${CONFIGURATIONS.FOLDER_CI_UID}
        |WHERE conf.${CONFIGURATIONS.CI_TYPE} IN (:types)""".stripMargin

  def findByTypes(ciTypes: Seq[String]): Seq[ConfigurationRow] =
    sqlQuery(STMT_FIND_BY_TYPE, params(
      "types" -> ciTypes.asJava
    ), configurationBinaryStreamRowMapper).toList

  def findFirstByTypes(ciTypes: Seq[String], folderId: Option[String]): Option[ConfigurationRow] = {
    val (query, mapper) = folderId match {
      case None => STMT_FIND_BY_TYPE_TITLE_IN_APPLICATIONS -> globalConfigurationBinaryStreamRowMapper
      case Some(_) => STMT_FIND_BY_TYPE_TITLE_AND_PARENT_FOLDERS -> configurationBinaryStreamRowMapper
    }
    findOne(
      sqlQuery(addLimitAndOffset(query, Some(1L), None), params(
        "types" -> ciTypes.asJava,
        "folderId" -> folderId.map(Ids.getName).orNull,
        "byTitle" -> false.asInteger,
        "title" -> null
      ), mapper)
    )
  }

  private val STMT_FIND_UIDS_BY_TYPE =
    s"""|SELECT
        |  conf.${CONFIGURATIONS.CI_UID}
        | FROM ${CONFIGURATIONS.TABLE} conf
        | WHERE conf.${CONFIGURATIONS.CI_TYPE} IN (:types)""".stripMargin

  def findUidsByTypes(ciTypes: Seq[String]): Seq[CiUid] = {
    sqlQuery(STMT_FIND_UIDS_BY_TYPE, params("types" -> ciTypes.asJava), rs => Integer.valueOf(rs.getInt(CONFIGURATIONS.CI_UID))).toSeq
  }

  def findAllConfigurationTypes: Seq[String] = sqlQuery(
    s"SELECT DISTINCT ${CONFIGURATIONS.CI_TYPE} FROM ${CONFIGURATIONS.TABLE}",
    params(),
    rs => rs.getString(CONFIGURATIONS.CI_TYPE)
  ).toSeq

  private val configurationBinaryStreamRowMapper: RowMapper[ConfigurationRow] = (rs: ResultSet, _: Int) => {
    val rawConfiguration = ResourceManagement.using(rs.getBinaryStream(CONFIGURATIONS.CONTENT))(decompress)
    mapFolderPath(rs) -> rawConfiguration
  }

  private val globalConfigurationBinaryStreamRowMapper: RowMapper[ConfigurationRow] = (rs: ResultSet, _: Int) => {
    None -> ResourceManagement.using(rs.getBinaryStream(CONFIGURATIONS.CONTENT))(decompress)
  }

  def deleteByTypes(ciTypes: Seq[String]): Try[Boolean] = {
    sqlExec(
      s"DELETE FROM ${CONFIGURATIONS.TABLE} WHERE ${CONFIGURATIONS.CI_TYPE} in (:ciTypes)",
      params("ciTypes" -> ciTypes.asJava),
      ps => Try(ps.execute())
    )
  }
}

class ConfigurationStoreException(msg: String, e: Throwable = null) extends RuntimeException(msg, e)
