package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.{Dependency, Release}
import com.xebialabs.xlrelease.repository.DependencyRepository
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.{DependencyPersistence, ReleasePersistence}
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

@IsTransactional
class SqlDependencyRepository(val releasePersistence: ReleasePersistence,
                              val dependencyPersistence: DependencyPersistence,
                              val repositoryAdapter: SqlRepositoryAdapter)
  extends DependencyRepository with BaseReleaseItemRepository with Logging {

  @Timed
  override def findById(id: String): Dependency = {
    val releaseId = releaseIdFrom(id)
    val release = getRelease(releaseId, id)
    val maybeDependency = release.getAllGates.asScala.flatMap(_.getDependencies.asScala).find(_.getId == id.normalized)
    maybeDependency.getOrElse(throw new NotFoundException(s"Repository entity [$id] not found"))
  }

  @Timed
  override def create(release: Release, dependency: Dependency): Dependency = {
    insertDependency(dependency)
    releasePersistence.update(release)
    dependency
  }

  @Timed
  override def update(release: Release, dependency: Dependency): Dependency = {
    if (!dependency.isArchived) {
      dependencyPersistence.updateDependency(dependency)
    }
    releasePersistence.update(release)
    dependency
  }

  @Timed
  override def update(release: Release, dependencies: Seq[Dependency]): Unit = {
    dependencies.foreach {  dependency =>
      if (!dependency.isArchived) {
        dependencyPersistence.updateDependency(dependency)
      }
    }
    releasePersistence.update(release)
  }

  @Timed
  override def delete(dependency: Dependency): Unit = {
    val release = dependency.getGateTask.getRelease
    deleteFromRelease(release, dependency)
  }

  private def deleteFromRelease(release: Release, dependency: Dependency): Unit = {
    val maybeGateTask = release.getAllGates.asScala.find(_.getId == dependency.getGateTask.getId.normalized)
    maybeGateTask.map { gateTask =>
      gateTask.setDependencies(
        gateTask.getDependencies.asScala.filterNot(_.getId == dependency.getId.normalized).asJava
      )
      releasePersistence.update(release)
      dependencyPersistence.deleteDependency(dependency)
      dependency
    }.getOrElse {
      throw new NotFoundException(
        s"GateTask ${dependency.getGateTask.getId} not found while deleting dependency ${dependency.getId}."
      )
    }
  }

  @Timed
  override def archive(release: Release, archivedDependencies: Seq[Dependency]): Unit = {
    if (archivedDependencies.nonEmpty) {
      archivedDependencies.foreach(dependencyPersistence.deleteDependency)
      releasePersistence.update(release)
    }
  }

  @Timed
  override def findAllIncomingDependencies(targetIds: Seq[String], statuses: Seq[String], referencingChildren: Boolean): Seq[Dependency] = {
    def status = if (statuses.isEmpty) None else Some(statuses.toSet)

    if (referencingChildren) {
      dependencyPersistence.findByPartialTargetIds(targetIds.toSet, status).asDependencies
    } else {
      dependencyPersistence.findByTargetIds(targetIds.toSet, status).asDependencies
    }
  }

}
