package com.xebialabs.xlrelease.repository.sql.proxy

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.repository.proxy.ResolvableConfigurationItemReference
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.serialization.json.repository.ResolverRepository
import grizzled.slf4j.Logging
import org.springframework.cglib.proxy.{MethodInterceptor, MethodProxy}

import java.lang.System.identityHashCode
import java.lang.reflect.Method
import java.util
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

case class ConfigurationItemCallback(referencedConfigurationItem: ResolvableConfigurationItemReference) extends MethodInterceptor with Logging {
  val id = referencedConfigurationItem.referencedId.normalized

  @volatile
  var isProxyInstanceInitialized: Boolean = false

  @volatile
  var preinitializedMethods: List[String] = List()

  @volatile
  var isInitialized: Boolean = false

  @volatile
  var isInitializationInProgress: Boolean = false

  @volatile
  var resolverRepository: ResolverRepository = _

  def doInitialize(enhancedInstance: AnyRef): Unit = {
    // accessing synthetic properties that are not stored in the database will initialize proxy
    // we don't want to initialize a proxy if:
    // - we're "pre-initializing" proxy (populating some of the properties from some DB table or JSON)
    // - we have already initializing the same proxy
    synchronized {
      if (isProxyInstanceInitialized && !isInitializationInProgress && !isInitialized) {
        logger.debug(s"Initializing $enhancedInstance")
        isInitializationInProgress = true
        val ci = enhancedInstance.asInstanceOf[ConfigurationItem]
        val readResult = Try(resolverRepository.read[ConfigurationItem](referencedConfigurationItem.referencedId, ci))
        readResult match {
          case Success(_) => ()
          case Failure(ex) =>
            val msg = s"Unable to fully initialize proxied item $referencedConfigurationItem"
            if (XlrConfig.getInstance.features.serialization.failOnUnresolvableReferences) {
              throw new IllegalStateException(msg, ex)
            } else {
              logger.warn(msg, ex)
            }
        }
        resolverRepository = null
        isInitialized = true
        isInitializationInProgress = false
      }
    }
  }

  def toString(enhancedInstance: AnyRef): CiId = {
    s"${identityHashCode(enhancedInstance)} ${referencedConfigurationItem.ciReference.toString}"
  }

  private def setResolverRepository(resolverRepository: ResolverRepository): AnyRef = {
    this.resolverRepository = resolverRepository
    null
  }

  def setProxyInstanceInitialized(isProxyInitialized: Boolean): AnyRef = {
    this.isProxyInstanceInitialized = isProxyInitialized
    null
  }

  def setPreinitializedMethods(methodNames: util.List[String]): AnyRef = {
    preinitializedMethods = methodNames.asScala.toList
    null
  }

  //noinspection ScalaStyle
  override def intercept(enhancedInstance: AnyRef, method: Method, args: Array[AnyRef], methodProxy: MethodProxy) = {
    method.getName match {
      case "setId" => methodProxy.invokeSuper(enhancedInstance, args)
      case "getId" => methodProxy.invokeSuper(enhancedInstance, args)
      case "setType" => methodProxy.invokeSuper(enhancedInstance, args)
      case "getType" => methodProxy.invokeSuper(enhancedInstance, args)
      case "hasProperty" => methodProxy.invokeSuper(enhancedInstance, args)
      case "toString" => toString(enhancedInstance)
      case "equals" => methodProxy.invokeSuper(enhancedInstance, args)
      case "hashCode" => methodProxy.invokeSuper(enhancedInstance, args)
      case "set$token" => methodProxy.invokeSuper(enhancedInstance, args)
      case "set$ciAttributes" => methodProxy.invokeSuper(enhancedInstance, args)
      case "set$resolverRepository" => setResolverRepository(args(0).asInstanceOf[ResolverRepository])
      case "set$proxyInstanceInitialized" => setProxyInstanceInitialized(args(0).asInstanceOf[Boolean])
      case "set$preinitializedMethods" => setPreinitializedMethods(args(0).asInstanceOf[java.util.List[String]])
      case "get$resolvableConfigurationItemReference" => referencedConfigurationItem
      case "isProxy" => true
      case "isInitialized" => isInitialized
      case "initialize" =>
        doInitialize(enhancedInstance)
        true
      case methodName if preinitializedMethods.contains(methodName) =>
        val res = methodProxy.invokeSuper(enhancedInstance, args)
        res
      case _ =>
        doInitialize(enhancedInstance)
        val res = methodProxy.invokeSuper(enhancedInstance, args)
        res
    }
  }
}
