package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.{ACTIVE_STATUSES, PLANNED}
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.TasksSqlBuilder.normalizeTags
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, PersistenceSupport}
import com.xebialabs.xlrelease.repository.{Ids, ReleaseRepository}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.PendingPlanItemScheduledJobService
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.jdbc.core.namedparam.{MapSqlParameterSource, SqlParameterSource}
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import java.util
import scala.collection.mutable
import scala.collection.parallel.CollectionConverters._
import scala.jdk.CollectionConverters._

@Component
class XLRelease970TablesUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                             @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                             @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                             releaseRepository: ReleaseRepository,
                                             pendingPlanItemScheduledJobService: PendingPlanItemScheduledJobService,
                                             xlrConfig: XlrConfig)
  extends Upgrade
    with PersistenceSupport
    with Logging
    with BatchSupport
    with TransactionSupport {

  private val batchSize: Int = Math.max(1, (xlrConfig.database.maxPoolSize - 2) / 2)

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.7.0#5")

  override def doUpgrade(): Boolean = {
    logger.info("Starting releases, tasks and task tags extraction upgrade")
    val releaseIds = releaseRepository.findIdsByStatus(ACTIVE_STATUSES :+ PLANNED: _*)
    val totalReleaseNum = releaseIds.length

    logger.info(s"Found $totalReleaseNum releases")

    doInBatch(releaseIds, batchSize = batchSize) { batch =>
      val items = if (xlrConfig.features.updates.parallel) {
        batch.items.par
      } else {
        batch.items
      }
      items.iterator.foreach { releaseId =>
        doInTransaction {
          upgradeRelease(releaseId)
        }
      }
    }
    logger.info("Finished releases, tasks and task tags extraction upgrade")
    true
  }

  def upgradeRelease(releaseId: String): Unit = {

    val release = releaseRepository.findById(releaseId, ResolveOptions.MINIMAL)

    //schedule the release autostart with quartz
    if (release.canScheduleReleaseStart) {
      pendingPlanItemScheduledJobService.schedule(release)
    }

    //Insert values for newly added column in XLR_RELEASES Table
    updateRelease(release.getCiUid, release)

    val allTasks = release.getAllTasks.asScala
    val taskUidMap: Map[String, CiUid] = buildTaskUidMap(release.getCiUid)

    val taskColumnParams = mutable.Buffer[SqlParameterSource]()
    val taskTagsDeleteParamSource = mutable.Buffer[SqlParameterSource]()
    val taskTagsInsertParamSource = mutable.Buffer[SqlParameterSource]()
    allTasks.foreach { task =>
      val taskUidOption = taskUidMap.get(Ids.getFolderlessId(task.getId))

      if (taskUidOption.isEmpty) {
        logger.warn(s"Unable to upgrade task: ${task.getId} for release: ${release.getTitle}, releaseCI_UID: ${release.getCiUid}")
      } else {
        val taskUid = taskUidOption.get

        //schedule the task autostart with quartz
        if (task.canScheduleTaskStart) {
          pendingPlanItemScheduledJobService.schedule(task)
        }

        //Insert values for newly added column in XLR_TASKS Table
        if (!task.getStatus.isDone && !task.getStatus.isDoneInAdvance) {
          taskColumnParams += taskUpdateParams(taskUid, task)
        }

        //Insert values in XLR_TASK_TAGS table
        if (task.getTags.asScala.nonEmpty) {
          taskTagsDeleteParamSource += taskTagsDeleteParams(taskUid)
          taskTagsInsertParamSource ++= taskTagsInsertParams(taskUid, task.getTags)
        }
      }
    }
    if (taskColumnParams.nonEmpty) {
      namedTemplate.batchUpdate(STMT_UPDATE_TASK, taskColumnParams.toArray)
    }
    if (taskTagsDeleteParamSource.nonEmpty) {
      namedTemplate.batchUpdate(STMT_DELETE_TASK_TAGS, taskTagsDeleteParamSource.toArray);
      namedTemplate.batchUpdate(STMT_INSERT_TASK_TAGS, taskTagsInsertParamSource.toArray);
    }
  }

  private val STMT_GET_TASK_UIDS =
    s"""|SELECT
        | ${TASKS.TASK_ID},
        | ${TASKS.CI_UID}
        |FROM ${TASKS.TABLE}
        |WHERE ${TASKS.RELEASE_UID} = :${TASKS.RELEASE_UID}
        |""".stripMargin

  private def buildTaskUidMap(releaseUid: CiUid): Map[String, CiUid] = {
    sqlQuery(STMT_GET_TASK_UIDS,
      params(TASKS.RELEASE_UID -> releaseUid),
      rs => rs.getString(TASKS.TASK_ID) -> CiUid(rs.getInt(TASKS.CI_UID))).toMap
  }

  private val STMT_DELETE_TASK_TAGS = s"DELETE FROM ${TASK_TAGS.TABLE} WHERE ${TASK_TAGS.CI_UID} = :${TASK_TAGS.CI_UID}"

  private def taskTagsDeleteParams(ciUid: CiUid): MapSqlParameterSource = {
    paramMap2MapSqlParameterSource(params(TASK_TAGS.CI_UID -> ciUid))
  }

  private val STMT_INSERT_TASK_TAGS =
    s"""INSERT INTO ${TASK_TAGS.TABLE} (${TASK_TAGS.CI_UID}, ${TASK_TAGS.VALUE})
       | VALUES(:${TASK_TAGS.CI_UID}, :${TASK_TAGS.VALUE})""".stripMargin

  private def taskTagsInsertParams(ciUid: CiUid, tags: util.List[String]): List[MapSqlParameterSource] = {
    val normalizedTags = normalizeTags(tags.asScala.toSet)
    normalizedTags.map { normalizedTag =>
      paramMap2MapSqlParameterSource(
        params(
          TASK_TAGS.CI_UID -> ciUid,
          TASK_TAGS.VALUE -> normalizedTag
        )
      )
    }.toList
  }

  //Only update the data in the new columns for XLR_RELEASE
  private val STMT_UPDATE_RELEASE =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET
        |   ${RELEASES.PLANNED_DURATION} = :plannedDuration,
        |   ${RELEASES.IS_OVERDUE_NOTIFIED} = :overdueNotified
        | WHERE ${RELEASES.CI_UID} = :ciUid""".stripMargin

  private def updateRelease(ciUid: CiUid, release: Release): Unit = {
    sqlUpdate(STMT_UPDATE_RELEASE, params(
      "plannedDuration" -> release.getPlannedDuration,
      "overdueNotified" -> release.isOverdueNotified.asInteger,
      "ciUid" -> ciUid
    ), _ => ())
  }

  //Only update the data in the new columns for XLR_TASKS
  private val STMT_UPDATE_TASK =
    s"""|UPDATE ${TASKS.TABLE}
        | SET
        |   ${TASKS.PLANNED_DURATION} = :plannedDuration,
        |   ${TASKS.IS_OVERDUE_NOTIFIED} = :overdueNotified,
        |   ${TASKS.IS_DUE_SOON_NOTIFIED} = :dueSoonNotified
        | WHERE ${TASKS.CI_UID} = :ciUid""".stripMargin

  private def taskUpdateParams(ciUid: CiUid, task: Task): MapSqlParameterSource = {
    paramMap2MapSqlParameterSource(
      params(
        "plannedDuration" -> task.getPlannedDuration,
        "overdueNotified" -> task.isOverdueNotified.asInteger,
        "dueSoonNotified" -> task.isDueSoonNotified.asInteger,
        "ciUid" -> ciUid
      )
    )
  }
}
