package com.xebialabs.xlrelease.scheduler.sql

import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.xlrelease.db.sql.SqlBuilder
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{RELEASES, TASK_JOBS}
import com.xebialabs.xlrelease.scheduler.JobRow
import com.xebialabs.xlrelease.scheduler.filters.JobFilters
import com.xebialabs.xlrelease.scheduler.sql.SqlJobFiltersUtil.jobFilter
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

object SqlJobFiltersQueryBuilder {
  def apply(dialect: SqlBuilder.Dialect, template: NamedParameterJdbcTemplate): SqlJobFiltersQueryBuilder = {
    new SqlJobFiltersQueryBuilder(dialect, template)
  }
}

object SqlJobFiltersUtil {
  def jobFilter[R](filter: JobFilters, qb: FilterQueryBuilderSupport[JobFilters, R])
  : FilterQueryBuilderSupport[JobFilters, R] = {
    if (filter.withoutNode) {
      qb.whereClauses += s"j.${TASK_JOBS.NODE} IS NULL"
    } else {
      if (isNotBlank(filter.node)) {
        qb.like(s"j.${TASK_JOBS.NODE}", "node", filter.node.trim)
      }
    }
    if (filter.status != null && !filter.status.isEmpty) {
      qb.whereClauses += s"j.${TASK_JOBS.STATUS} IN (:status)"
      qb.queryParams += "status" -> filter.status
    }
    if (isNotBlank(filter.releaseId)) {
      qb.joinClauses +=
        s"""
           | JOIN ${RELEASES.TABLE} r ON r.${RELEASES.CI_UID} = j.${TASK_JOBS.RELEASE_UID}
           |""".stripMargin
      qb.whereClauses += s"r.${RELEASES.RELEASE_ID} = :releaseId"
      qb.queryParams += "releaseId" -> Ids.getName(filter.releaseId.trim)
    }
    qb
  }
}

class SqlJobFiltersQueryBuilder(val dialect: Dialect, val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[JobFilters, JobRow] with FilterQueryBuilderSupport[JobFilters, JobRow] with JobRowMapper {

  private[sql] val STMT_SELECT =
    s"""SELECT
       | j.${TASK_JOBS.ID},
       | j.${TASK_JOBS.TASK_ID},
       | j.${TASK_JOBS.RELEASE_UID},
       | j.${TASK_JOBS.EXECUTION_ID},
       | j.${TASK_JOBS.JOB_TYPE},
       | j.${TASK_JOBS.STATUS},
       | j.${TASK_JOBS.VERSION},
       | j.${TASK_JOBS.SUBMIT_TIME},
       | j.${TASK_JOBS.SCHEDULED_START_TIME},
       | j.${TASK_JOBS.START_TIME},
       | j.${TASK_JOBS.NODE},
       | j.${TASK_JOBS.RUNNER_ID}
       | FROM ${TASK_JOBS.TABLE} j
       |""".stripMargin

  private lazy val queryTemplate =
    s"""
        $STMT_SELECT
       | $joinClause
       | $whereClause
       | $orderClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val totalQueryTemplate =
    s"""
       | SELECT COUNT(1)
       | FROM ${TASK_JOBS.TABLE} j
       | $joinClause
       | $whereClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  def from(filter: JobFilters): FiltersQueryBuilder[JobFilters, JobRow] = {
    jobFilter(filter, this)
    this
  }

  override def build(): PageableQuery[JobRow] = {
    remapSortOrderParams()
    buildSortOrderClause("scheduledStartTime")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[JobRow](namedTemplate, resultsQueryString, queryParams.toMap, jobMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[JobRow](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      "id" -> TASK_JOBS.ID,
      "node" -> TASK_JOBS.NODE,
      "status" -> TASK_JOBS.STATUS,
      "submitTime" -> TASK_JOBS.SUBMIT_TIME,
      "scheduledStartTime" -> TASK_JOBS.SCHEDULED_START_TIME,
      "startTime" -> TASK_JOBS.START_TIME,
      "releaseUid" -> TASK_JOBS.RELEASE_UID
    )
  }
}
