package com.xebialabs.xlrelease.scheduler.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TASK_LOGS
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence.hash
import com.xebialabs.xlrelease.scheduler.TaskLogRow.taskLogRowMapper
import com.xebialabs.xlrelease.scheduler.TaskLogRow
import com.xebialabs.xlrelease.scheduler.repository.TaskLogRepository
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate

@IsTransactional
class SqlTaskLogRepository(implicit val jdbcTemplate: JdbcTemplate, val dialect: Dialect)
  extends TaskLogRepository
    with PersistenceSupport
    with Logging
    with Utils {

  @Timed
  @IsReadOnly
  override def read(taskId: String, executionId: String): Option[TaskLogRow] = {
    findOptional(_.queryForObject(STMT_FIND_TASK_EXECUTION, taskLogRowMapper, hash(taskId), executionId))
  }

  @Timed
  @IsReadOnly
  override def read(taskId: String): Seq[TaskLogRow] = {
    findMany(sqlQuery(STMT_FIND_ALL_TASK_EXECUTIONS, params(TASK_LOGS.TASK_ID_HASH -> hash(taskId)), taskLogRowMapper))
  }

  @Timed
  override def create(row: TaskLogRow): TaskLogRow = {
    namedTemplate.update(STMT_INSERT_ROW, row.asSqlParamMap())
    row
  }

  @Timed
  override def update(row: TaskLogRow): TaskLogRow = {
    namedTemplate.update(STMT_UPDATE_DATES, row.asSqlParamMap())
    row
  }

  @Timed
  override def delete(taskId: String, executionId: String): Unit = {
    val params = Map(
      TASK_LOGS.TASK_ID_HASH -> hash(taskId),
      TASK_LOGS.EXECUTION_ID -> executionId
    )
    namedTemplate.update(STMT_DELETE_ROW, params)
  }

  private val STMT_INSERT_ROW =
    s"""INSERT INTO ${TASK_LOGS.TABLE} (
       |  ${TASK_LOGS.TASK_ID_HASH},
       |  ${TASK_LOGS.EXECUTION_ID},
       |  ${TASK_LOGS.LAST_MODIFIED_DATE},
       |  ${TASK_LOGS.END_DATE}
       | ) VALUES (
       |  :${TASK_LOGS.TASK_ID_HASH},
       |  :${TASK_LOGS.EXECUTION_ID},
       |  :${TASK_LOGS.LAST_MODIFIED_DATE},
       |  :${TASK_LOGS.END_DATE}
       | )
       |""".stripMargin

  private val STMT_UPDATE_DATES =
    s"""UPDATE ${TASK_LOGS.TABLE}
       |  SET
       |    ${TASK_LOGS.LAST_MODIFIED_DATE} = :${TASK_LOGS.LAST_MODIFIED_DATE},
       |    ${TASK_LOGS.END_DATE} = :${TASK_LOGS.END_DATE}
       |  WHERE
       |    ${TASK_LOGS.TASK_ID_HASH} = :${TASK_LOGS.TASK_ID_HASH}
       |  AND
       |    ${TASK_LOGS.EXECUTION_ID} = :${TASK_LOGS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_DELETE_ROW =
    s"""DELETE FROM ${TASK_LOGS.TABLE}
       |  WHERE
       |    ${TASK_LOGS.TASK_ID_HASH} = :${TASK_LOGS.TASK_ID_HASH}
       |  AND
       |    ${TASK_LOGS.EXECUTION_ID} = :${TASK_LOGS.EXECUTION_ID}
       |""".stripMargin

  private val STMT_FIND_TASK_EXECUTION =
    s"""SELECT * FROM ${TASK_LOGS.TABLE}
       |  WHERE ${TASK_LOGS.TASK_ID_HASH} = ?
       |  AND ${TASK_LOGS.EXECUTION_ID} = ?
       |""".stripMargin

  private val STMT_FIND_ALL_TASK_EXECUTIONS =
    s"""SELECT * FROM ${TASK_LOGS.TABLE}
       |  WHERE ${TASK_LOGS.TASK_ID_HASH} = :${TASK_LOGS.TASK_ID_HASH}
       |  ORDER BY ${TASK_LOGS.LAST_MODIFIED_DATE} ASC
       |""".stripMargin
}
