package com.xebialabs.xlrelease.repository.sql.artifact.resolution

import com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver
import com.xebialabs.deployit.engine.spi.artifact.resolution.{ArtifactResolver, ResolvedArtifactFile}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.JdbcTemplateHolder
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{ARTIFACTS, RELEASES}
import org.springframework.jdbc.datasource.DataSourceUtils
import org.springframework.jdbc.support.JdbcUtils

import java.io.InputStream
import java.net.URI

object SqlArtifactResolver {
  val Protocol = "sql"

  def canHandle(uri: String): Boolean = URI.create(uri).getScheme == Protocol
}


@Resolver(protocols = Array("sql"))
class SqlArtifactResolver extends ArtifactResolver {

  override def resolveLocation(artifact: SourceArtifact): ResolvedArtifactFile = new ResolvedArtifactFile {

    override def getFileName: String = artifact.getName

    override def close(): Unit = {
    }

    override def openStream(): InputStream = getArtifactInputStream(artifact.getId)

    private def getArtifactInputStream(id: String): InputStream = new InputStream {
      private val GET_ARTIFACT_CONTENT: String =
        s"""SELECT
           | ${ARTIFACTS.CONTENT}
           |  FROM ${ARTIFACTS.TABLE}
           |  WHERE ${ARTIFACTS.ARTIFACTS_ID} = ?
           |  AND ${ARTIFACTS.CI_UID} =
           |     (SELECT ${RELEASES.CI_UID}
           |     FROM ${RELEASES.TABLE}
           |     WHERE ${RELEASES.RELEASE_ID} = ?)""".stripMargin

      private val datasource = JdbcTemplateHolder.get().getDataSource
      private val con = DataSourceUtils.getConnection(datasource)

      private val ps = con.prepareStatement(GET_ARTIFACT_CONTENT)
      ps.setString(1, getReleaselessChildId(id))
      ps.setString(2, getName(releaseIdFrom(id)))

      private val rs = ps.executeQuery()
      if (!rs.next()) throw new NotFoundException(s"Artifact data not found for: $id")

      private val delegate = rs.getBinaryStream(ARTIFACTS.CONTENT)

      override def read(): Int = delegate.read()

      override def read(b: Array[Byte], off: Int, len: Int): Int = delegate.read(b, off, len)

      override def close(): Unit = {
        JdbcUtils.closeResultSet(rs)
        JdbcUtils.closeStatement(ps)
        DataSourceUtils.releaseConnection(con, datasource)
      }
    }
  }

  override def validateCorrectness(artifact: SourceArtifact): Boolean = getFileUri(artifact).getScheme == SqlArtifactResolver.Protocol

  private def getFileUri(artifact: SourceArtifact) = URI.create(artifact.getFileUri)

}
