package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.{Release, ReleaseExtension}
import com.xebialabs.xlrelease.json.JsonUtils.DashboardJsonOps
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.ReleaseExtensionsRepository.GenericReleaseExtensionRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.repository.sql.persistence.data.ReleaseExtensionRow
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._


@IsTransactional
class SqlReleaseExtensionsRepository(val releaseExtensionPersistence: ReleaseExtensionPersistence,
                                     val repositoryAdapter: SqlRepositoryAdapter,
                                     releasePersistence: ReleasePersistence)
  extends GenericReleaseExtensionRepository
    with Logging {

  @Timed
  override def exists(extensionId: String): Boolean = {
    logger.debug(s"exists($extensionId)")
    releaseExtensionPersistence.exists(extensionId).nonEmpty
  }

  @Timed
  override def read(extensionId: String): Option[ReleaseExtension] = {
    logger.debug(s"read($extensionId)")
    for {
      releaseUid <- releasePersistence.findUidByReleaseId(releaseIdFrom(extensionId))
      extensionRow <- releaseExtensionPersistence.read(releaseUid, extensionId)
      extension <- repositoryAdapter.deserialize[ReleaseExtension](extensionRow.json.withoutUnknownTiles)
    } yield {
      extension.setId(extensionId)
      extension
    }
  }

  @Timed
  override def readAll(releaseId: String): Seq[ReleaseExtension] = {
    logger.debug(s"readAll($releaseId)")
    for {
      releaseUid <- releasePersistence.findUidByReleaseId(releaseId).toList
      extensionRow <- releaseExtensionPersistence.readAll(releaseUid)
      extension <- repositoryAdapter.deserialize[ReleaseExtension](extensionRow.json.withoutUnknownTiles)
    } yield {
      extension.setId(ReleaseExtensionRow.toExternalId(releaseId, extension.getName))
      extension
    }
  }

  @Timed
  override def create(extension: ReleaseExtension): ReleaseExtension = {
    logger.debug(s"create(${extension.getId})")
    withRow(extension)(releaseExtensionPersistence.create)
    extension
  }

  @Timed
  override def createAll(release: Release, extensionsIds: Set[String]): Int = {
    logger.debug(s"createAll(${release.getId}, $extensionsIds)")
    val extensionsNames = extensionsIds.map(ReleaseExtensionRow.toInternalId)
    releaseExtensionPersistence.createAll(
      release.getCiUid,
      release.getExtensions.asScala
        .map(ReleaseExtensionRow.fromExtension)
        .filter(ext => extensionsNames contains ext.id)
        .toSeq
    )
  }

  @Timed
  override def update(extension: ReleaseExtension): Boolean = {
    logger.debug(s"update(${extension.getId})")
    withRow(extension)(releaseExtensionPersistence.update) == 1
  }

  @Timed
  override def delete(extensionId: String): Boolean = {
    logger.debug(s"delete($extensionId)")
    withRelease(releaseIdFrom(extensionId))(releaseExtensionPersistence.delete(_, extensionId)) > 0
  }

  @Timed
  def createAll(release: Release): Seq[ReleaseExtension] = {
    logger.debug(s"createAll(${release.getId})")
    val extensions = release.getExtensions.asScala.toSeq
    releaseExtensionPersistence.createAll(release.getCiUid, extensions.map(ReleaseExtensionRow.fromExtension))
    extensions
  }

  @Timed
  def deleteAll(releaseId: String): Int = {
    logger.debug(s"deleteAll($releaseId)")
    withRelease0(releaseId)(releaseExtensionPersistence.deleteAll).getOrElse(0)
  }

  protected def withRow[A, T <: ReleaseExtension](extension: T)(f: (CiUid, ReleaseExtensionRow) => A): A =
    withRow0(extension)(f).getOrElse {
      throw notFound(releaseIdFrom(extension.getId))
    }

  protected def withRow0[A, T <: ReleaseExtension](extension: T)(f: (CiUid, ReleaseExtensionRow) => A): Option[A] = {
    withRelease0(releaseIdFrom(extension.getId)) { uid =>
      f(uid, ReleaseExtensionRow.fromExtension(extension))
    }
  }

  protected def withRelease0[A](releaseId: CiId)(f: CiUid => A): Option[A] =
    releasePersistence.findUidByReleaseId(releaseIdFrom(releaseId)).map(f)

  protected def withRelease[A](releaseId: CiId)(f: CiUid => A): A =
    withRelease0(releaseId)(f).getOrElse {
      throw notFound(releaseId)
    }

  protected def notFound(releaseId: String) =
    new NotFoundException(s"Release [$releaseId] not found.")

}
