package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.domain.Release
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, JdbcTemplate, RowMapper}
import org.springframework.util.StringUtils

import java.nio.charset.CodingErrorAction
import java.nio.charset.StandardCharsets.UTF_8
import java.nio.{ByteBuffer, CharBuffer}
import java.sql.{PreparedStatement, ResultSet, Timestamp}
import java.time.Instant
import java.util.Date
import scala.collection.mutable
import scala.util.Try

trait Utils {

  def findOptional[A](body: JdbcTemplate => A)(implicit jdbcTemplate: JdbcTemplate): Option[A] = {
    try {
      Option(body(jdbcTemplate))
    } catch {
      case _: EmptyResultDataAccessException => None
    }
  }

  def findOne[A](body: => mutable.Buffer[A]): Option[A] =
    Try(body)
      .map(_.headOption)
      .recover { case _: EmptyResultDataAccessException => None }
      .get

  def findMany[A](body: => mutable.Buffer[A]): Seq[A] =
    Try(body)
      .map(_.toSeq)
      .recover { case _: EmptyResultDataAccessException => Seq() }
      .get

}

object Utils {

  def params(map: (String, _ <: Any)*): Map[String, Any] = map.toMap[String, Any]

  def rowMapper[A](f: ResultSet => A): RowMapper[A] = (rs: ResultSet, _: Int) => f(rs)

  def processSCMData(release: Release, planItemProcessFunction: Release => _, scmTraceabilityDeletionFunc: Integer => Boolean): Unit = {
    val scmId = Option(release.get$ciAttributes().getScmTraceabilityDataId)
    release.get$ciAttributes().setScmTraceabilityDataId(null)

    planItemProcessFunction(release)

    scmId.foreach(scmTraceabilityDeletionFunc(_))
  }

  implicit class RichBooleanAsInt(val b: Boolean) extends AnyVal {
    def asInteger: Integer = Integer.valueOf(if (b) 1 else 0)
  }

  implicit class RichJBooleanAsInt(val b: java.lang.Boolean) extends AnyVal {
    def asInteger: Integer = Integer.valueOf(if (b) 1 else 0)
  }

  implicit class RichIntAsBoolean(val i: Int) extends AnyVal {
    def asBoolean: Boolean = i != 0
  }

  implicit class BatchJdbc(val jdbcTemplate: JdbcTemplate) extends AnyVal {
    def batch(sql: String, id: Integer, batch: List[String]): Unit = {
      jdbcTemplate.batchUpdate(sql, new BatchPreparedStatementSetter {
        override def getBatchSize: Int = batch.size

        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setInt(1, id)
          ps.setString(2, batch(i))
        }
      })
    }
  }

  implicit class RichDateAsTimestamp(val date: Date) extends AnyVal {
    def asTimestamp: Timestamp =
      Option(date).map(d => new Timestamp(d.getTime)).orNull
  }

  implicit class RichInstantAsTimestamp(val instant: Instant) extends AnyVal {
    def asTimestamp: Timestamp =
      Option(instant).map(d => Timestamp.from(d)).orNull
  }

  implicit class RichStringAsTruncatable(val value: String) extends AnyVal {
    def trimAndTruncate(size: Int): String = StringUtils.trimWhitespace(value).truncate(size)

    def truncate(size: Int): String =
      Option(value).map(s =>
        s.substring(0, if (s.length > size) size else s.length)
      ).orNull

    def truncateBytes(size: Int): String = {
      Option(value).map(s => {
        val bytes = s.getBytes(UTF_8)
        if (bytes.length <= size) {
          s
        } else {
          val decoder = UTF_8.newDecoder()
          decoder.onMalformedInput(CodingErrorAction.IGNORE)
          decoder.reset()
          val bb = ByteBuffer.wrap(bytes, 0, size)
          val cb = CharBuffer.allocate(value.length)
          decoder.decode(bb, cb, true)
          decoder.flush(cb)

          new String(cb.array(), 0, cb.position())
        }
      }).orNull
    }
  }

}
