package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthenticatedUserName, getAuthentication}
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.ReleaseGlobalAndFolderVariablesDecorator.GLOBAL_AND_FOLDER_VARIABLES
import com.xebialabs.xlrelease.api.internal.{DecoratorsCache, InternalMetadataDecoratorService}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.sql.SqlRepositoryAdapter
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.{CommentPersistence, ReleasePersistence, ReleasesSqlBuilder, TaskPersistence}
import com.xebialabs.xlrelease.service.TaskSearchService.DEFAULT_TASK_LIMIT
import com.xebialabs.xlrelease.views.converters.TasksViewConverter
import com.xebialabs.xlrelease.views.{ReleaseTasks, TaskSearchView, TasksFilters}
import grizzled.slf4j.Logging

import java.util.{List => JList}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

class SqlTaskSearchService(
                            roleService: RoleService,
                            taskAccessService: TaskAccessService,
                            tasksViewConverter: TasksViewConverter,
                            decoratorService: InternalMetadataDecoratorService,
                            releasePersistence: ReleasePersistence,
                            taskPersistence: TaskPersistence,
                            commentPersistence: CommentPersistence,
                            val repositoryAdapter: SqlRepositoryAdapter,
                            implicit val permissionEnforcer: PermissionEnforcer,
                            implicit val sqlDialect: Dialect)
  extends TaskSearchService
    with Logging {

  private val MAX_NUMBER_OF_JOINS = 5

  @Timed
  override def getTasksByRelease(tasksFilters: TasksFilters, limitTasksHintOption: Integer): TaskSearchView = {
    val limit: Int = Option(limitTasksHintOption).map(_.intValue()).getOrElse(DEFAULT_TASK_LIMIT)

    val taskIds: Seq[CiId] = time("finding task ids") {
      searchTaskIds(tasksFilters, limit)
    }

    val releaseIds = taskIds.map(releaseIdFrom).distinct

    logger.debug(s"Found ${taskIds.size} tasks in ${releaseIds.size} releases for user $getAuthenticatedUserName")

    val allowedTaskTypesForAuthenticatedUser = time("fetching allowed task types for authenticated user") {
      taskAccessService.getAllowedTaskTypesForAuthenticatedUser
    }

    val releasesTasks: Seq[ReleaseTasks] = if (releaseIds.nonEmpty) {
      time("total time to create releasesTasks") {
        val decoratorsCache = new DecoratorsCache()
        val queryReleases = new ReleasesSqlBuilder().selectReleaseData().withReleaseIds(releaseIds).build()
        val releases = time("de-serializing releases") {
          releasePersistence.findReleaseDatasByQuery(queryReleases).map(repositoryAdapter.tryDeserialize)
        }.flatten

        time("decorating releases") {
          decoratorService.decorate(releases.asJava, Seq(GLOBAL_AND_FOLDER_VARIABLES, EFFECTIVE_SECURITY).asJava, decoratorsCache)
        }
        val commentsByTasks = commentPersistence.countByTasks(releases.map(_.getCiUid))

        releases.map { release =>
          val orderedTasks = release.getAllTasks.asScala
            .filter(task => taskIds.exists(foundId => task.getId.endsWith(foundId)))
            .sortBy(task => taskIds.indexWhere(id => task.getId.startsWith(id)))
            .map { task =>
              val fullView = tasksViewConverter.toFullView(task, allowedTaskTypesForAuthenticatedUser)
              fullView.setNumberOfComments(commentsByTasks.getOrElse(Ids.getFolderlessId(fullView.getId), 0))
              fullView
            }
          new ReleaseTasks(release, orderedTasks.asJava)
        }
      }
    } else {
      Seq.empty
    }

    new TaskSearchView(sortByReleaseTitle(releasesTasks), taskIds.size >= limit)
  }


  private def time[A](desc: String)(block: => A): A = {
    val startTime = System.currentTimeMillis()
    val res = block
    val endTime = System.currentTimeMillis()
    val total = endTime - startTime
    logger.trace(s"TIME: $desc took $total")
    res
  }

  private def sortByReleaseTitle(releasesTasks: Seq[ReleaseTasks]): JList[ReleaseTasks] = {
    releasesTasks.sortBy(releaseTasks => Option(releaseTasks.getTitle).getOrElse("")).asJava
  }

  //  @VisibleForTesting
  def searchTaskIds(tasksFilters: TasksFilters, limit: Int): Seq[CiId] = {
    def getPrincipalsFromGroup: Iterable[Either[String, String]] => Seq[String] = _.collect {
      case Left(id) => id
    }.toSeq

    def getRoleIdsFromGroup: Iterable[Either[String, String]] => Seq[String] = _.collect {
      case Right(id) => id
    }.toSeq

    def doSearch(principals: Seq[String], roleIds: Seq[String]): Seq[CiId] = {
      logger.info(s"${tasksFilters}: Searching with ${principals.length} principals and ${roleIds.length} roles")
      val tasksQuery = SqlTasksFilterSupport.sqlBuilderByFilters(tasksFilters, principals, roleIds)
        .limit(limit)
        .build()
      taskPersistence.findTaskIdsByQuery(tasksQuery)
    }

    def getPrincipals = authenticationToPrincipals(getAuthentication).asScala.map(Left.apply[String, String])

    def getRoleIds = roleService.getRolesFor(getAuthentication).asScala.map(_.getId).map(Right.apply[String, String])

    val taskIds = (getPrincipals ++ getRoleIds)
      .grouped(sqlDialect.maxQueryParams / MAX_NUMBER_OF_JOINS)
      .foldLeft(mutable.LinkedHashSet.empty[CiId]) {
        case (acc, group) =>
          if (acc.size < limit) {
            val newBatch = doSearch(getPrincipalsFromGroup(group), getRoleIdsFromGroup(group))
            acc ++ newBatch
          } else {
            acc
          }
      }.take(limit)
    logger.debug(s"${tasksFilters}: results ${taskIds.size}")
    taskIds.toSeq
  }

}
