package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.builder.TaskBuilder.newCustomScript
import com.xebialabs.xlrelease.customscripts.ScriptTypes
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.PythonScript.{CUSTOM_SCRIPT_TASK_PROPERTY, PYTHON_SCRIPT_PROPERTY}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.TaskUpdatedEvent
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.utils.TaskTypes
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.sql.persistence.DependencyPersistence
import com.xebialabs.xlrelease.repository.{CiProperty, Ids, TaskRepository}
import com.xebialabs.xlrelease.service.SqlTaskTypeConversion._
import org.springframework.stereotype.Component

import java.util
import scala.jdk.CollectionConverters._

object SqlTaskTypeConversion {
  val TYPE_CUSTOMSCRIPTTASK: Type = Type.valueOf(classOf[CustomScriptTask])
  val TYPE_TASK: Type = Type.valueOf(classOf[Task])
  val TYPE_TASK_GROUP: Type = Type.valueOf(classOf[TaskGroup])
}

@IsTransactional
@Component
class SqlTaskTypeConversion(scriptTypes: ScriptTypes,
                            taskRepository: TaskRepository,
                            eventBus: XLReleaseEventBus,
                            dependencyPersistence: DependencyPersistence
                           ) extends TaskTypeConversion {

  @Timed
  override def changeActiveTaskType(taskId: String, taskOrScriptType: Type): Task = {
    val newTypeIsPythonScriptTask = scriptTypes.getPythonScriptTypes.contains(taskOrScriptType)

    var newType: Type = taskOrScriptType
    var scriptType: Type = null

    if (newTypeIsPythonScriptTask) {
      newType = SqlTaskTypeConversion.TYPE_CUSTOMSCRIPTTASK
      scriptType = taskOrScriptType
    }

    val oldTask: Task = taskRepository.findById(taskId)
    checkTaskIsPlanned(oldTask)

    val newTypeIsTaskInstance = newType.instanceOf(TYPE_TASK)
    checkArgument(newTypeIsPythonScriptTask || newTypeIsTaskInstance,
      s"Cannot change type of task '${oldTask.getTitle}' to '$newType' because it is not one of supported types: ${getAllSupportedTypes.mkString(",")}")

    val oldType = oldTask.getType
    if (oldType == newType && (scriptType == null || oldTask.getTaskType == scriptType)) {
      return oldTask
    }

    if (oldType.isSubTypeOf(TYPE_TASK_GROUP)) {
      throw new IllegalArgumentException(s"Conversion of task '${oldTask.getTitle}' from '$oldType' is not supported.")
    }

    var excludedProperties: Seq[String] = Seq.empty
    var updatedTask: Task = null

    if (newType.instanceOf(TYPE_CUSTOMSCRIPTTASK)) {
      checkArgument(scriptType != null, "Script type must be provided if new task type is CustomScriptTask")
      updatedTask = newCustomScript(scriptType.toString).withId(oldTask.getId).build
      excludedProperties = excludedProperties :+ PYTHON_SCRIPT_PROPERTY
      copyPythonScriptProperties(oldTask, updatedTask.asInstanceOf[CustomScriptTask])
    }
    else {
      updatedTask = Task.fromType(newType)
    }
    updatedTask.setId(oldTask.getId)
    copyNonContainmentProperties(oldTask, updatedTask, excludedProperties)
    removeVariableMappingsForNonExistingProperties(oldTask, updatedTask)

    moveContainedCollectionProperties(oldTask, updatedTask)
    updateLinks(oldTask, updatedTask)
    updateInContainer(oldTask, updatedTask)
    updatedTask.setTaskFailureHandlerEnabled(updatedTask.isTaskFailureHandlerEnabled)
    updateGateTaskDependencies(oldTask)
    // S-91304: task type is changed, but there is no real need to update 'task-types' cache on all nodes
    val res = taskRepository.updateType(updatedTask)
    eventBus.publish(TaskUpdatedEvent(oldTask, res))

    res
  }


  private def checkTaskIsPlanned(task: Task): Unit = {
    checkArgument(task.isPlanned, "Cannot change type of the task '%s' because it is in state %s and not %s", task.getTitle, task.getStatus, TaskStatus.PLANNED)
  }

  private def getAllSupportedTypes: Seq[Type] = {
    TaskTypes.getDefaultTaskTypes.asScala ++ scriptTypes.getPythonScriptTypes.asScala ++ scriptTypes.getContainerTaskTypes.asScala
  }.toSeq

  private def copyPythonScriptProperties(oldTask: Task, updatedTask: CustomScriptTask): Unit = {
    val newPythonScript = updatedTask.getPythonScript
    if (oldTask.getType.instanceOf(TYPE_CUSTOMSCRIPTTASK)) {
      val propertySource = oldTask.asInstanceOf[CustomScriptTask].getPythonScript
      copyNonContainmentProperties(propertySource, newPythonScript,
        Seq(CUSTOM_SCRIPT_TASK_PROPERTY) ++ PythonScript.UPDATEABLE_PROPERTIES.asScala)
    }
  }

  private def copyNonContainmentProperties(from: ConfigurationItem, to: ConfigurationItem, excludedProperties: Seq[String]): Unit = {
    for (newPd <- to.getType.getDescriptor.getPropertyDescriptors.asScala) {
      if (!excludedProperties.contains(newPd.getName) && !newPd.isHidden) {
        val oldPd = from.getType.getDescriptor.getPropertyDescriptor(newPd.getName)
        if (isSameKindOfProperty(newPd, oldPd) && hasSameCategory(newPd, oldPd)) {
          if (oldPd.getKind.isSimple) {
            newPd.set(to, oldPd.get(from))
          } else if (!oldPd.isAsContainment && hasCopyableReferencedType(oldPd, newPd)) {
            newPd.set(to, oldPd.get(from))
          }
        }
      }
    }
  }

  private def removeVariableMappingsForNonExistingProperties(oldTask: Task, updatedTask: Task) = {
    updatedTask.getVariableMapping.keySet.removeIf((fqPropertyName: String) => {
      val newProperty = CiProperty.of(updatedTask, fqPropertyName)
      val oldProperty = CiProperty.of(oldTask, fqPropertyName)
      !oldProperty.isPresent || !newProperty.isPresent || !isSameKindOfProperty(oldProperty.get.getDescriptor, newProperty.get.getDescriptor)
    })
  }

  private def moveContainedCollectionProperties(from: ConfigurationItem, to: ConfigurationItem): Unit = {
    for (newPd <- to.getType.getDescriptor.getPropertyDescriptors.asScala) {
      val oldPd = from.getType.getDescriptor.getPropertyDescriptor(newPd.getName)
      if (isSameKindOfProperty(newPd, oldPd)) {
        val isContainedCollectionProperty = oldPd.isAsContainment && (oldPd.getKind ne PropertyKind.CI)
        if (isContainedCollectionProperty && hasCopyableReferencedType(oldPd, newPd)) {
          val referencedType: Type = oldPd.getReferencedType
          if (null != referencedType) {
            val oldCollection = oldPd.get(from).asInstanceOf[util.Collection[ConfigurationItem]]
            oldCollection.forEach({ ci =>
              val newId: String = to.getId + "/" + Ids.getName(ci.getId)
              ci.setId(newId)
            })
            newPd.set(to, oldCollection)
          }
        }
      }
    }
  }

  private def updateLinks(oldTask: Task, newTask: Task): Unit = {
    oldTask.getContainer match {
      case group: ParallelGroup =>
        for (link <- group.getLinksOf(oldTask).asScala) {
          if (link.hasTarget(oldTask)) {
            link.setTarget(newTask)
          }
          if (link.hasSource(oldTask)) {
            link.setSource(newTask)
          }
        }
      case _ =>
    }
  }

  private def updateInContainer(oldTask: Task, newTask: Task): Unit = {
    oldTask.getContainer.replaceTask(oldTask, newTask)
  }

  private def updateGateTaskDependencies(task: Task): Unit = task match {
    case gateTask: GateTask => gateTask.getDependencies.asScala.foreach(dependencyPersistence.deleteDependency)
    case _ =>
  }

  private def isSameKindOfProperty(newPd: PropertyDescriptor, oldPd: PropertyDescriptor) = {
    oldPd != null && oldPd.getKind == newPd.getKind && oldPd.isPassword == newPd.isPassword && oldPd.isAsContainment == newPd.isAsContainment
  }

  private def hasSameCategory(newPd: PropertyDescriptor, oldPd: PropertyDescriptor) = {
    oldPd != null && oldPd.getCategory == newPd.getCategory
  }

  private def hasCopyableReferencedType(oldPd: PropertyDescriptor, newPd: PropertyDescriptor) = {
    val oldReferencedType = oldPd.getReferencedType
    val newReferencedType = newPd.getReferencedType
    (oldReferencedType == null && newReferencedType == null) || (oldReferencedType != null && oldReferencedType.instanceOf(newReferencedType))
  }

}
