package com.xebialabs.xlrelease.scheduler.sql

import com.xebialabs.xlrelease.db.sql.SqlBuilder
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{TASKS, TASK_JOBS}
import com.xebialabs.xlrelease.scheduler.JobOverview
import com.xebialabs.xlrelease.scheduler.filters.JobFilters
import com.xebialabs.xlrelease.scheduler.sql.SqlJobFiltersUtil.jobFilter
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

object SqlJobOverviewQueryBuilder {
  def apply(dialect: SqlBuilder.Dialect, template: NamedParameterJdbcTemplate): SqlJobOverviewQueryBuilder = {
    new SqlJobOverviewQueryBuilder(dialect, template)
  }
}

class SqlJobOverviewQueryBuilder(val dialect: Dialect, val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[JobFilters, JobOverview]
    with FilterQueryBuilderSupport[JobFilters, JobOverview]
    with JobOverviewRowMapper {

  private[sql] val STMT_SELECT_WITH_TASK_TYPE =
    s"""SELECT
       | j.${TASK_JOBS.ID},
       | j.${TASK_JOBS.TASK_ID},
       | j.${TASK_JOBS.JOB_TYPE},
       | j.${TASK_JOBS.STATUS},
       | j.${TASK_JOBS.SUBMIT_TIME},
       | j.${TASK_JOBS.SCHEDULED_START_TIME},
       | j.${TASK_JOBS.START_TIME},
       | j.${TASK_JOBS.NODE},
       | j.${TASK_JOBS.RUNNER_ID},
       | t.${TASKS.TASK_TYPE}
       | FROM ${TASK_JOBS.TABLE} j
       |""".stripMargin

  private lazy val queryTemplate =
    s"""
        $STMT_SELECT_WITH_TASK_TYPE
       | $joinClause
       | $whereClause
       | $orderClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val totalQueryTemplate =
    s"""
       | SELECT COUNT(1)
       | FROM ${TASK_JOBS.TABLE} j
       | $joinClause
       | $whereClause
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  def from(filter: JobFilters): FiltersQueryBuilder[JobFilters, JobOverview] = {
    jobFilter(filter, this)
    joinTaskType()
    this
  }

  private def joinTaskType(): Unit = {
    joinClauses +=
      s"""
         | JOIN ${TASKS.TABLE} t ON t.${TASKS.TASK_ID_HASH} = j.${TASK_JOBS.TASK_ID_HASH}
         |""".stripMargin
  }

  override def build(): PageableQuery[JobOverview] = {
    remapSortOrderParams()
    buildSortOrderClause("scheduledStartTime")
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[JobOverview](namedTemplate, resultsQueryString, queryParams.toMap,
      jobOverviewMapper)
    val totalCountQuery = new SqlQuery[Long](namedTemplate, totalQueryTemplate, queryParams.toMap, (rs, _) => rs
      .getLong(1))
    new SqlPageableQuery[JobOverview](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      "id" -> TASK_JOBS.ID,
      "node" -> TASK_JOBS.NODE,
      "status" -> TASK_JOBS.STATUS,
      "submitTime" -> TASK_JOBS.SUBMIT_TIME,
      "scheduledStartTime" -> TASK_JOBS.SCHEDULED_START_TIME,
      "startTime" -> TASK_JOBS.START_TIME
    )
  }

}
