package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.blackout.BlackoutMetadata.toStartOfNextMinute
import com.xebialabs.xlrelease.domain.calendar.Blackout
import com.xebialabs.xlrelease.domain.status.TaskStatus.PENDING
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.utils.syntax._
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.query.TaskBasicData
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import grizzled.slf4j.Logging

import java.util
import java.util.{Date, List => JList, Map => JMap}
import scala.annotation.varargs
import scala.jdk.CollectionConverters._

@IsTransactional
class SqlTaskRepository(val releasePersistence: ReleasePersistence,
                        val dependencyPersistence: DependencyPersistence,
                        val taskPersistence: TaskPersistence,
                        val commentRepository: CommentRepository,
                        val repositoryAdapter: SqlRepositoryAdapter,
                        val configurationPersistence: ConfigurationReferencePersistence,
                        val facetRepositoryDispatcher: FacetRepositoryDispatcher,
                        implicit val dialect: Dialect)
  extends TaskRepository
    with BaseReleaseItemRepository
    with DeserializationSupport
    with ConfigurationReferencesSupport
    with Logging {

  @Timed
  override def findById[T <: Task](taskId: String): T = {
    this.findById(taskId, ResolveOptions.WITH_DECORATORS)
  }

  @Timed
  override def findById[T <: Task](taskId: String, resolveOptions: ResolveOptions): T = {
    if (!exists(taskId)) {
      throw new NotFoundException(s"Repository entity [$taskId] not found")
    }
    val release = getRelease(releaseIdFrom(taskId), taskId, resolveOptions)
    Option(release.getTask(taskId.normalized).asInstanceOf[T])
      .fold(throw new NotFoundException(s"Repository entity [$taskId] not found in release")) { task =>
        task.getAllTasks.forEach { t => commentRepository.decorate(t) }
        task
      }
  }

  @Timed
  @IsReadOnly
  override def findTasksBasicData(taskIds: JList[String]): JList[TaskBasicData] = if (taskIds.isEmpty) {
    Seq.empty.asJava
  } else {
    val tasksData = taskPersistence.findByIds(taskIds.asScala)
    // iterate based on requested task IDs to avoid having to find
    // actual full task IDs with folders and release
    taskIds.asScala
      .map(taskId => (taskId, tasksData.find(data => taskId.endsWith(data.taskId))))
      .flatMap {
        case (id, Some(data)) => Some(data.asTaskData(id))
        case _ => None
      }
      .asJava
  }

  @Timed
  @IsReadOnly
  override def exists(taskId: String): Boolean = {
    logger.debug(s"Checking if exists task [$taskId]")
    taskPersistence.exists(taskId)
  }

  @Timed
  override def update(task: Task): Task = {
    val release = task.getRelease
    releasePersistence.update(release)
    updateTaskProperties(task)
    updateConfigurationRefs(release)
    task
  }

  @Timed
  @varargs
  override def updateTasks(tasks: Task*): Unit = {
    tasks.groupBy(_.getRelease).foreach {
      case (release, tasks) =>
        releasePersistence.update(release)
        tasks.foreach { task =>
          updateTaskProperties(task)
        }
        updateConfigurationRefs(release)
    }
  }

  @Timed
  override def updateTaskProperties(task: Task): Unit = {
    taskPersistence.updateProperties(task)
    task.dependencies.foreach {
      case dep if dep.isArchived =>
        dependencyPersistence.deleteDependency(dep)
      case dep =>
        try {
          dependencyPersistence.updateDependency(dep)
        } catch {
          case e: NotFoundException =>
            logger.error(s"Dependency $dep not found when updating task $task", e)
        }
    }
  }

  @Timed
  override def batchUpdateTaskProperties(tasks: Set[Task]): Unit = {
    taskPersistence.batchUpdateTaskProperties(tasks)
    val (archivedDeps, otherDeps) = tasks.flatMap(_.dependencies).partition(_.isArchived)
    dependencyPersistence.batchDeleteDependencies(archivedDeps)
    dependencyPersistence.batchUpdateDependencies(otherDeps)
  }

  @Timed
  override def updateTaskAndReleaseFlagStatus(task: Task, release: Release): Task =
    update(task)

  @Timed
  override def updateType(task: Task): Task = {
    val updated = update(task)
    taskPersistence.updateType(task)
    updated
  }

  @Timed
  override def delete(task: Task): Unit = {
    checkIsNotReferencedByDependencies(task.getId)
    releasePersistence.update(task.getRelease)
    task.allDependencies.foreach(dependencyPersistence.deleteDependency)
    task.getAllTasks.forEach { t =>
      commentRepository.deleteByTask(t)
      taskPersistence.delete(t)
    }
    updateConfigurationRefs(task.getRelease)
  }

  private def createTask(newTask: Task): Task = {
    taskPersistence.insert(newTask, newTask.getRelease.getCiUid)
    if (newTask.isInstanceOf[TaskGroup]) {
      newTask.getAllTasks.asScala
        .filter(_.getContainer.getId == newTask.getId)
        .map(createTask)
    }
    newTask
  }

  @Timed
  override def create(newTask: Task): Task = {
    releasePersistence.update(newTask.getRelease)
    val createdTask = createTask(newTask)
    createdTask.allDependencies.foreach(insertDependency)
    updateConfigurationRefs(createdTask.getRelease)
    facetRepositoryDispatcher.liveRepository.createFromTasks(newTask.getAllTasks.asScala.toSeq)
    createdTask
  }

  @Timed
  override def moveTask(taskToMove: Task,
                        movedTask: Task,
                        updatedOriginContainer: TaskContainer,
                        updatedTargetContainer: TaskContainer): Task = {
    if (updatedOriginContainer != updatedTargetContainer) {
      checkIsNotReferencedByDependencies(taskToMove.getId)

      taskToMove.getAllTasks.asScala.foreach { taskWithOldId =>
        val oldId = taskWithOldId.getId
        val newId = oldId.replace(taskToMove.getId, movedTask.getId)
        taskPersistence.move(oldId, newId)
      }
    }
    releasePersistence.update(movedTask.getRelease)
    movedTask
  }

  @Timed
  override def findAffectedByBlackout(blackout: Blackout): JList[String] = {
    val endDate = toStartOfNextMinute(blackout.getEndDate)

    val sqlWithParams = new ReleasesSqlBuilder()
      .selectReleaseData()
      .withOneOfStatuses(ReleaseStatus.ACTIVE_STATUSES)
      .build()
    releasePersistence
      .findReleaseDatasByQuery(sqlWithParams)
      .flatMap(tryDeserializeRelease)
      .flatMap(rel => rel.getAllTasks.asScala)
      .filter(task => task.getStatus == TaskStatus.PENDING && task.isPostponedDueToBlackout && task.getScheduledStartDate == endDate)
      .map(_.getId).asJava
  }

  @Timed
  override def findPendingTasksWithFacets(): JList[String] = {
    val sqlWithParams: (String, Seq[AnyRef]) = new TasksSqlBuilder()
      .selectTaskId()
      .withOneOfTaskStatuses(Seq(PENDING))
      .withAnyFacets()
      .build()
    taskPersistence
      .findTaskIdsByQuery(sqlWithParams)
      .toSet // withAnyFacets joins XLR_FACETS which may produce duplicate XLR_TASKS records
      .toList
      .asJava
  }

  @Timed
  @IsReadOnly
  override def getTitle(taskId: String): String =
    taskPersistence.getTitle(taskId)
      .getOrElse(throw new NotFoundException("Task [%s] not found", taskId))

  @Timed
  @IsReadOnly
  override def getAllTags(limitNumber: Int): util.Set[String] = {
    taskPersistence.findAllTags(limitNumber).asJava
  }

  @Timed
  @IsReadOnly
  override def findOverdueTaskIds(): Seq[String] = {
    taskPersistence.findOverdueTaskIds()
  }

  @Timed
  @IsReadOnly
  override def findDueSoonTaskIds(): Seq[String] = {
    taskPersistence.findDueSoonTaskIds()
  }

  @Timed
  @IsReadOnly
  override def getStatus(taskId: String): TaskStatus = {
    taskPersistence.getStatus(taskId).getOrElse {
      throw new LogFriendlyNotFoundException(s"getStatus: could not find task '${taskId}'")
    }
  }

  @Timed
  @IsReadOnly
  override def getTaskStatuses(releaseId: String): JMap[String, TaskStatus] = {
    taskPersistence.getTaskStatuses(releaseId)
  }

  @Timed
  @IsReadOnly
  override def findTaskIdsByTaskTypeStatusAndStartDate(taskType: Type, taskStatus: TaskStatus, startedBefore: Date): Seq[String] = {
    taskPersistence.findTaskIdsByTaskTypeStatusAndStartDate(taskType, taskStatus, startedBefore)
  }
}
