package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlplatform.utils.ResourceManagement
import com.xebialabs.xlrelease.builder.CommentBuilder
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.{Comment, Release, Task}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{SEPARATOR, getFolderlessId, getName}
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence.CommentRow.fromComment
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{COMMENTS, RELEASES, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.dao.DataIntegrityViolationException
import org.springframework.jdbc.core.namedparam.{MapSqlParameterSource, SqlParameterSource}
import org.springframework.jdbc.core.support.SqlLobValue
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.Types
import scala.jdk.CollectionConverters._
import scala.util.Try

@IsTransactional
class CommentPersistence(val jdbcTemplate: JdbcTemplate,
                         val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with Logging with Utils with LimitOffset {


  @IsReadOnly
  def exists(taskUid: CiUid, commentId: CiId): Boolean = {
    findOne {
      sqlQuery(STMT_EXISTS, params("taskUid" -> taskUid, "commentId" -> getName(commentId)), _.getInt(1) == 1)
    }.getOrElse(false)
  }

  def batchCreate(taskCommentRows: Set[TaskCommentRow]): Unit = {
    val batchArgs: Array[SqlParameterSource] = taskCommentRows.collect {
      case TaskCommentRow(taskUid, CommentRow(commentId, author, created, modified, comment)) =>
        val sqlParameterSource = new MapSqlParameterSource()
        sqlParameterSource.addValue("taskUid", taskUid)
        sqlParameterSource.addValue("commentId", getName(commentId))
        sqlParameterSource.addValue("author", author)
        sqlParameterSource.addValue("creationDate", created.toDate)
        sqlParameterSource.addValue("modifiedDate", modified.toDate)
        sqlParameterSource.addValue("content", new SqlLobValue(compress(comment)), Types.BLOB)
        sqlParameterSource
    }.toArray
    namedTemplate.batchUpdate(STMT_CREATE, batchArgs)
  }

  def create(taskUid: CiUid, row: CommentRow): Boolean = {
    logger.debug(s"create($taskUid, $row)")
    Try {
      sqlExecWithContent(STMT_CREATE,
        params(
          "taskUid" -> taskUid,
          "commentId" -> getName(row.commentId),
          "author" -> row.author,
          "creationDate" -> row.created.toDate,
          "modifiedDate" -> row.modified.toDate
        ),
        "content" -> row.comment,
        _ == 1
      )
    }.recover {
      case _: DataIntegrityViolationException => false
    }.get
  }

  def createAll(taskUid: CiUid, rows: Seq[CommentRow]): Int = {
    logger.debug(s"createAll($taskUid, ${rows.map(_.commentId)}")
    Try {
      sqlBatchWithContent(STMT_CREATE,
        rows.map(row => params(
          "taskUid" -> taskUid,
          "commentId" -> getName(row.commentId),
          "author" -> row.author,
          "creationDate" -> row.created.toDate,
          "modifiedDate" -> row.modified.toDate
        ) -> ("content" -> row.comment)
        ))
    }.recover {
      case _: DataIntegrityViolationException => Seq.empty
    }.get.sum
  }

  @IsReadOnly
  def read(taskUid: CiUid, commentId: CiId): Option[CommentRow] = {
    logger.debug(s"read($taskUid, $commentId)")
    findOne {
      sqlQuery(STMT_READ_ONE, params("taskUid" -> taskUid, "commentId" -> getName(commentId)), commentRowMapper)
    }
  }

  @IsReadOnly
  def readAll(taskUid: CiUid): Seq[CommentRow] = {
    logger.debug(s"readAll($taskUid)")
    sqlQuery(STMT_READ, params("taskUid" -> taskUid), commentRowMapper).toSeq
  }

  def update(taskUid: CiUid, commentId: CiId, content: String, modified: Option[DateTime] = None): Boolean = {
    logger.debug(s"update($taskUid, $commentId, $content, $modified)")
    sqlExecWithContent(STMT_UPDATE,
      params(
        "taskUid" -> taskUid,
        "commentId" -> getName(commentId),
        "modifiedDate" -> modified.getOrElse(DateTime.now).toDate
      ),
      "newContent" -> content,
      _ == 1
    )
  }

  def delete(taskUid: CiUid, commentId: CiId): Int = {
    logger.debug(s"delete($taskUid, $commentId)")
    sqlUpdate(STMT_DELETE_BY_ID, params("taskUid" -> taskUid, "commentId" -> getName(commentId)), identity)
  }

  def deleteAll(taskUid: CiUid): Int = {
    logger.debug(s"deleteAll($taskUid)")
    sqlUpdate(STMT_DELETE, params("taskUid" -> taskUid), identity)
  }

  def deleteByRelease(releaseUid: CiUid, taskCiUids: Seq[CiUid]): Int = {
    logger.debug(s"deleteByRelease($releaseUid)")
    taskCiUids.grouped(STMT_DELETE_BY_RELEASE_BATCH_SIZE).map(batch => {
      sqlUpdate(STMT_DELETE_BY_RELEASE, params("taskCiUids" -> batch.map(_.asInstanceOf[Integer]).asJava), identity)
    }).sum
  }

  @IsReadOnly
  def decorate(taskUid: CiUid, task: Task): Task = {
    task.getComments.clear()
    val comments = readAll(taskUid).map(_.toComment(task.getId))
    logger.trace(s"decorating task ${Ids.getName(task.getId)} with ${comments.size} comments")
    task.getComments.addAll(comments.asJava)
    task
  }

  @IsReadOnly
  def decorate(release: Release): Release = {
    logger.debug(s"decorate(${release.getId})")
    release.clearComments()
    sqlQuery(STMT_READ_BY_RELEASE, params("releaseUid" -> release.getCiUid), releaseCommentsDecorator(release))
    release
  }

  @IsReadOnly
  def countByTasks(releaseCiUids: Seq[CiUid]): Map[String, Int] = if (releaseCiUids.nonEmpty) {
    logger.debug(s"countByTasks($releaseCiUids)")
    sqlQuery(STMT_COUNT_BY_TASK_ID, params("releaseUids" -> releaseCiUids.asJava), (rs, _) => {
      (rs.getString(TASKS.TASK_ID), rs.getInt("COUNT"))
    }).toMap
  } else {
    Map.empty
  }

  private val commentRowMapper: RowMapper[CommentRow] = (rs, _) => {
    val content = ResourceManagement.using(rs.getBinaryStream(COMMENTS.CONTENT))(decompress)
    CommentRow(
      commentId = rs.getString(COMMENTS.COMMENT_ID),
      author = rs.getString(COMMENTS.AUTHOR),
      created = new DateTime(rs.getTimestamp(COMMENTS.CREATION_DATE)),
      modified = new DateTime(rs.getTimestamp(COMMENTS.MODIFIED_DATE)),
      comment = content
    )
  }

  private def releaseCommentsDecorator(release: Release): RowMapper[Unit] = {
    logger.debug(s"releaseCommentsDecorator(${release.getId})")
    val tasks = release.getAllTasks.asScala.map { task =>
      getFolderlessId(task.getId) -> task
    }.toMap
    (rs, i) => {
      val row = commentRowMapper.mapRow(rs, -1)
      val taskId = rs.getString(TASKS.TASK_ID)
      logger.trace(s"[$i] adding comment ${row.commentId} to task $taskId")
      tasks.get(taskId) match {
        case None =>
          logger.warn(s"[$i] Task [$taskId] for comment [${row.commentId}] not found in release [${release.getId}].")
        case Some(task) =>
          logger.trace(s"[$i] Task [$taskId] <- [${row.commentId}]@${task.getComments.size}")
          task.getComments.add(row.toComment(release.getId + Ids.SEPARATOR + Ids.getReleaselessChildId(taskId)))
      }
    }
  }
}

object CommentPersistence {

  case class CommentRow(commentId: String, author: String, created: DateTime, modified: DateTime, comment: String)

  case class TaskCommentRow(taskUid: Int, commentRow: CommentRow)

  object CommentRow {
    def fromComment(comment: Comment): CommentRow = CommentRow(
      commentId = getName(comment.getId),
      author = comment.getAuthor,
      created = new DateTime(comment.getCreationDate),
      modified = new DateTime(comment.getDate),
      comment = comment.getText
    )
  }

  object TaskCommentRow {
    def fromTask(task: Task): Seq[TaskCommentRow] = {
      task.getComments.asScala.map(c => TaskCommentRow(task.getCiUid, fromComment(c))).toSeq
    }
  }

  implicit class CommentRowOps(val commentRow: CommentRow) extends AnyVal {
    def toComment(taskId: String): Comment = {
      CommentBuilder.newComment()
        .withId(taskId + SEPARATOR + commentRow.commentId)
        .withAuthor(commentRow.author)
        .withCreationDate(commentRow.created.toDate)
        .withDate(commentRow.modified.toDate)
        .withText(commentRow.comment)
        .build
    }
  }

  private val STMT_EXISTS: String =
    s"""SELECT 1 FROM ${COMMENTS.TABLE} WHERE ${COMMENTS.TASK_UID} = :taskUid AND ${COMMENTS.COMMENT_ID} = :commentId"""

  private val STMT_CREATE: String =
    s"""|INSERT INTO ${COMMENTS.TABLE} (
        |   ${COMMENTS.TASK_UID},
        |   ${COMMENTS.COMMENT_ID},
        |   ${COMMENTS.AUTHOR},
        |   ${COMMENTS.CREATION_DATE},
        |   ${COMMENTS.MODIFIED_DATE},
        |   ${COMMENTS.CONTENT}
        |) VALUES (
        |   :taskUid,
        |   :commentId,
        |   :author,
        |   :creationDate,
        |   :modifiedDate,
        |   :content
        |)
     """.stripMargin


  private val STMT_SELECT: String =
    s"""|SELECT
        |   c.${COMMENTS.TASK_UID},
        |   c.${COMMENTS.COMMENT_ID},
        |   c.${COMMENTS.AUTHOR},
        |   c.${COMMENTS.CREATION_DATE},
        |   c.${COMMENTS.MODIFIED_DATE},
        |   c.${COMMENTS.CONTENT}
        |FROM  ${COMMENTS.TABLE} c
     """.stripMargin

  private val STMT_READ: String =
    STMT_SELECT +
      s"""|WHERE c.${COMMENTS.TASK_UID} = :taskUid
          |ORDER BY
          |  c.${COMMENTS.CREATION_DATE} ASC,
          |  c.${COMMENTS.COMMENT_ID} ASC
       """.stripMargin

  private val STMT_READ_ONE: String =
    STMT_SELECT +
      s"""|WHERE c.${COMMENTS.TASK_UID} = :taskUid
          |  AND c.${COMMENTS.COMMENT_ID} = :commentId""".stripMargin

  private val STMT_COUNT_BY_TASK_ID: String =
    s"""|SELECT
        |   t.${TASKS.TASK_ID},
        |   COUNT(c.${COMMENTS.COMMENT_ID}) AS COUNT
        |FROM  ${COMMENTS.TABLE} c
        |JOIN ${TASKS.TABLE} t on t.${TASKS.CI_UID} = c.${COMMENTS.TASK_UID}
        |JOIN ${RELEASES.TABLE} r on r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
        |WHERE r.${RELEASES.CI_UID} IN (:releaseUids)
        |GROUP BY t.${TASKS.TASK_ID}
     """.stripMargin

  private val STMT_READ_BY_RELEASE: String =
    s"""|SELECT
        |   t.${TASKS.CI_UID},
        |   t.${TASKS.TASK_ID},
        |   c.${COMMENTS.COMMENT_ID},
        |   c.${COMMENTS.AUTHOR},
        |   c.${COMMENTS.CREATION_DATE},
        |   c.${COMMENTS.MODIFIED_DATE},
        |   c.${COMMENTS.CONTENT}
        |FROM  ${COMMENTS.TABLE} c
        |JOIN ${TASKS.TABLE} t on t.${TASKS.CI_UID} = c.${COMMENTS.TASK_UID}
        |JOIN ${RELEASES.TABLE} r on r.${RELEASES.CI_UID} = t.${TASKS.RELEASE_UID}
        |WHERE r.${RELEASES.CI_UID} = :releaseUid
        |ORDER BY
        |   t.${TASKS.TASK_ID} ASC,
        |   c.${COMMENTS.CREATION_DATE} ASC,
        |   c.${COMMENTS.COMMENT_ID} ASC
     """.stripMargin

  private val STMT_UPDATE: String =
    s"""|UPDATE ${COMMENTS.TABLE}
        |SET
        |   ${COMMENTS.MODIFIED_DATE} = :modifiedDate,
        |   ${COMMENTS.CONTENT} = :newContent
        |WHERE
        |   ${COMMENTS.TASK_UID} = :taskUid AND
        |   ${COMMENTS.COMMENT_ID} = :commentId
     """.stripMargin

  private val STMT_DELETE: String =
    s"""|DELETE FROM ${COMMENTS.TABLE}
        |WHERE  ${COMMENTS.TASK_UID} = :taskUid
     """.stripMargin

  private val STMT_DELETE_BY_ID: String =
    STMT_DELETE + s"""AND ${COMMENTS.COMMENT_ID} = :commentId"""

  private val STMT_DELETE_BY_RELEASE: String =
    s"""|DELETE FROM ${COMMENTS.TABLE}
        |WHERE ${COMMENTS.TASK_UID} IN (:taskCiUids)""".stripMargin

  private val STMT_DELETE_BY_RELEASE_BATCH_SIZE: Integer = 512
}
