package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.booter.local.utils.Strings.isEmpty
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.FolderVariablesDecorator.FOLDER_VARIABLES
import com.xebialabs.xlrelease.api.internal.{DecoratorsCache, EffectiveSecurity, InternalMetadataDecoratorService}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, ReleasePersistence}
import com.xebialabs.xlrelease.security.XLReleasePermissions.{AUDIT_ALL, EDIT_FOLDER, VIEW_FOLDER}
import com.xebialabs.xlrelease.utils.Tree.Node
import grizzled.slf4j.Logging
import org.springframework.jdbc.core._

import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@IsTransactional
class SqlFolderRepository(folderPersistence: FolderPersistence,
                          releasePersistence: ReleasePersistence,
                          teamRepository: TeamRepository,
                          permissionEnforcer: PermissionEnforcer,
                          roleService: RoleService,
                          val decoratorService: InternalMetadataDecoratorService,
                          implicit val jdbcTemplate: JdbcTemplate,
                          implicit val dialect: Dialect)
  extends FolderRepository
    with InterceptedRepository[Folder]
    with SqlRepository
    with Logging {

  @Timed
  @IsReadOnly
  override def exists(folderId: String): Boolean = {
    folderPersistence.exists(folderId)
  }

  @Timed
  @IsReadOnly
  override def getTitle(folderId: String): Option[String] = {
    findById(folderId, depth = 0).map(_.getTitle)
  }

  @Timed
  @IsReadOnly
  override def getPath(folderId: String): Seq[String] = {
    folderPersistence.getFolderPathSegments(folderId)
  }

  @Timed
  @IsReadOnly
  def getUid(folderId: String): CiUid = {
    folderPersistence.getUid(folderId)
  }

  @Timed
  override def findById(folderId: String, depth: Int): Option[Folder] =
    findNodeById(folderId, depth)
      .toOption
      .map { tree =>
        val folder = tree.toFolder
        decoratorService.decorate(folder, Seq(EFFECTIVE_SECURITY, FOLDER_VARIABLES).asJava)
        folder
      }

  @Timed
  override def findViewableFoldersById(folderId: String, depth: Int, enforcePermission: Boolean = true): Option[Folder] = {
    val cache = new DecoratorsCache()
    findViewableNodesById(folderId, depth, enforcePermission)
      .toOption
      .map { tree =>
        val folder = tree.toFolder
        decoratorService.decorate(folder, Seq(EFFECTIVE_SECURITY, FOLDER_VARIABLES).asJava, cache)
        folder.setChildren(decorateWithEffectiveSecurity(folder.getChildren.asScala.toSeq, cache).toSet.asJava)
        folder
      }
  }

  @Timed
  override def listViewableFolders(parentId: String, page: Page, decorateWithPermissions: Boolean = true, enforcePermission: Boolean = true): Seq[Folder] = {
    val viewableResults = paginate(listViewableNodes(parentId, page.depth.intValue(), enforcePermission).map(_.toFolder), page)
    if (decorateWithPermissions) {
      val decoratorsCache = new DecoratorsCache()
      decorateWithEffectiveSecurity(viewableResults, decoratorsCache)
    } else {
      // even we don't want permissions we still need view and edit folder to be used in folders page
      // getting all permissions is very costly operation
      val editableResults = findEditableFoldersById(parentId, page.depth.intValue())
      decorateWithViewPermission(viewableResults)
      decorateWithEditPermission(viewableResults, editableResults.map(_.asFolder.getId).toSeq)
      viewableResults
    }
  }

  @Timed
  override def findNumberOfChildrenForAFolder(folderId: String): Int = {
    folderPersistence.findNumberOfChildrenForAFolder(folderId);
  }

  private def decorateWithEditPermission(folders: Seq[Folder], foldersWithEdit: Seq[String]): Unit = {
    folders.foreach(f => {
      if (foldersWithEdit.contains(f.getId)) {
        val effectiveSecurity = f.get$metadata().get("security").asInstanceOf[EffectiveSecurity]
        effectiveSecurity.getPermissions.add(EDIT_FOLDER.getPermissionName)
      }
      decorateWithEditPermission(f.getChildren.asScala.toSeq, foldersWithEdit)
    })
  }

  private def decorateWithViewPermission(folders: Seq[Folder]): Unit = {
    folders.foreach(f => {
      val permissions = new java.util.HashSet[String]
      permissions.add(VIEW_FOLDER.getPermissionName)
      f.get$metadata().put("security", new EffectiveSecurity(permissions, java.util.Set.of()))
      decorateWithViewPermission(f.getChildren.asScala.toSeq)
    })
  }

  @Timed
  def setAsEffectiveSecurityId(folderId: String): Unit = {
    folderPersistence.setAsEffectiveSecuredCi(folderId)
  }

  @Timed
  def inheritEffectiveSecurityId(folderId: String): Unit = {
    folderPersistence.inheritEffectiveSecuredCi(folderId)
  }

  @Timed
  override def create(parentId: String, folder: Folder): Folder = {
    Try {
      interceptCreate(folder)
      insertNode(folder.getTitle, folder.getId, Some(parentId))
    } match {
      case Success(created) => created.toFolder
      case Failure(e) =>
        logger.error(e)
        throw new FoldersStoreException(s"Cannot create folder ${folder.getId} under $parentId")
    }
  }

  @Timed
  override def delete(folderId: String, deleteRelease: (Int, String) => Unit): Unit = {
    val ciUid = findNodeById(folderId, 0).value.uid
    folderPersistence.deleteReleases(ciUid, deleteRelease)
    findNodeById(folderId).bottomUp.foreach { data =>
      if (data.hasSecurity) {
        teamRepository.deleteTeamsFromPlatform(data.folderId.absolute)
      }
      folderPersistence.deleteByUid(data.uid)
      interceptDelete(data.folderId.absolute)
    }
  }

  @Timed
  override def rename(folderId: String, newName: String): Folder = {
    folderPersistence.rename(folderId, newName)
  }

  @Timed
  override def move(folderId: String, newParentId: String): Folder = {
    val oldSubtreeSecurityUid = findNodeById(folderId, depth = 0).value.securityUid
    val moved = folderPersistence.move(folderId, newParentId)
    val newSubtreeSecurityUid = moved.value.securityUid
    releasePersistence.replaceSecurityUid(moved.value.uid, oldSubtreeSecurityUid, newSubtreeSecurityUid)
    moved.toFolder
  }

  @Timed
  override def findByPath(titlePath: String, depth: Int = Int.MaxValue): Folder = {
    if (isEmpty(titlePath)) {
      throw new NotFoundException(s"Cannot find folder by empty path: $titlePath")
    }
    val pathFromRoot = if (titlePath.startsWith(Ids.SEPARATOR)) titlePath.substring(1) else titlePath

    logger.debug(s"Finding folder by path $pathFromRoot")

    val pathElements = pathFromRoot.split(Ids.SEPARATOR).filterNot(_.isEmpty)

    if (pathElements.isEmpty) {
      throw new IllegalArgumentException("No folder path specified, the root path is not supported")
    }

    val topLevelData: Option[FolderRow] = pathElements.headOption match {
      case None =>
        throw new NotFoundException(s"Cannot find folder by empty path: $titlePath")
      case Some(topName) =>
        findSubFolderDataByTitle(ROOT_FOLDER_ID, topName)
    }
    val topLevelUid = topLevelData.map(_.uid).getOrElse {
      throw new NotFoundException(s"Could not find folder [${pathElements.headOption.getOrElse("")}] in path [$titlePath]")
    }

    val topNode = findNodeByUid(topLevelUid, Math.max(depth, pathElements.tail.length)).toOption.get

    pathElements.tail.foldLeft(topNode) {
      case (node, pathElement) =>
        node.children.find(_.value.name == pathElement).getOrElse {
          throw new NotFoundException(s"Could not find folder [$pathElement] in path [$titlePath]")
        }
    }.toFolder
  }

  @Timed
  override def findSubFolderByTitle(parentId: String, name: String): Option[Folder] = findSubFolderDataByTitle(parentId, name).map(_.asFolder)

  @Timed
  @IsReadOnly
  override def isFolderInherited(folderId: String): Boolean = {
    folderPersistence.isInherited(folderId)
  }

  private[sql] def findSubFolderDataByTitle(folderId: String, name: String): Option[FolderRow] = {
    folderPersistence.findSubFolderByTitle(name, folderId)
  }

  private[sql] def insertNode(name: String, givenId: String, parentIdOpt: Option[String]): Node[FolderRow] = {
    folderPersistence.create(name, givenId, parentIdOpt)
  }

  private[sql] def findNodeById(folderId: String, depth: Int = Int.MaxValue): Node[FolderRow] = {
    folderPersistence.findById(folderId, depth)
  }

  private[sql] def findNodeByUid(ciUid: CiUid, depth: Int = Int.MaxValue): Node[FolderRow] = {
    val viewableTree = {
      if (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
        folderPersistence.findByUid(ciUid, depth)
      } else {
        folderPersistence.findByUidHavingPermission(ciUid,
          VIEW_FOLDER,
          authenticationToPrincipals(getAuthentication).asScala,
          getUserRoles(),
          depth)
      }
    }
    viewableTree
  }

  private def listViewableNodes(folderId: String, depth: Int = Int.MaxValue, enforcePermission: Boolean = true): List[Node[FolderRow]] = {
    val viewableTree: Node[FolderRow] = findViewableNodesById(folderId, depth, enforcePermission)
    viewableTree.toOption.fold(List.empty[Node[FolderRow]])(_.children)
  }

  private[sql] def findViewableNodesById(folderId: String, depth: Int = Int.MaxValue, enforcePermission: Boolean = true): Node[FolderRow] = {
    val viewableTree = {
      if (!enforcePermission || (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL))) {
        folderPersistence.findById(folderId, depth)
      } else {
        folderPersistence.findByIdHavingPermission(folderId,
          VIEW_FOLDER,
          authenticationToPrincipals(getAuthentication).asScala,
          getUserRoles(),
          depth)
      }
    }
    viewableTree
  }

  // we need flat folders here, if we build the tree nodes without parent they will be dropped
  private[sql] def findEditableFoldersById(folderId: String, depth: Int = Int.MaxValue): mutable.Buffer[FolderRow] = {
    val folders = {
      if (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
        folderPersistence.findDescendantsById(folderId, depth)
      } else {
        folderPersistence.findDescendantsByIdHavingPermission(folderId,
          EDIT_FOLDER,
          authenticationToPrincipals(getAuthentication).asScala,
          getUserRoles(),
          depth)
      }
    }
    folders
  }

  private def getUserRoles() = {
    roleService.getRolesFor(getAuthentication).asScala.map(_.getId)
  }
}
