package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, PATHS, VIEW}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.rowMapper
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow.{FolderEntry, Root}
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.jdbc.core.RowMapper


object FindDescendantsQueryBuilder extends SecurableSqlBuilder {
  private val selectDescendants: String =
    s"""
       | SELECT
       |   d.${VIEW.FOLDER_ID},
       |   d.${VIEW.FOLDER_PATH},
       |   d.${VIEW.DESCENDANT_UID},
       |   d.${VIEW.SECURITY_UID},
       |   d.${VIEW.TOKEN},
       |   d.${VIEW.NAME},
       |   p.${PATHS.ANCESTOR_UID},
       |   d.${VIEW.DEPTH}
       | FROM ${FOLDERS.TABLE} f
       """.stripMargin

  private val descendantsOf: String =
    s"""
       | JOIN ${VIEW.TABLE} d
       |    ON ( d.${VIEW.ANCESTOR_UID} = f.${FOLDERS.CI_UID} AND d.${PATHS.DEPTH} <= :depth )
       """.stripMargin

  // parentOf (or root for itself)
  private val parentOf: String =
    s"""
       | JOIN ${PATHS.TABLE} p
       |    ON  ( p.${PATHS.DESCENDANT_UID} = d.${VIEW.DESCENDANT_UID}
       |          AND (     p.${PATHS.DEPTH} = 1
       |               OR ( p.${PATHS.DEPTH} = 0 AND p.${PATHS.ANCESTOR_UID} = ${Root.uid} ) ) )
       """.stripMargin

  private def findAncestorWithPermission(folderCondition: String, hasRoles: Boolean): String =
    s"""
       | WHERE
       |    $folderCondition
       | AND (
       |            f.${FOLDERS.FOLDER_ID} = d.${VIEW.FOLDER_ID}
       |        OR (
       |              d.${VIEW.SECURITY_UID} IN (
       |                  ${selectCiIdsWithPermission(hasRoles)}
       |              )
       |           )
       |     )
     """.stripMargin

  private val folderIdCondition: String = s"f.${FOLDERS.FOLDER_ID} = :folderId"

  private val folderUidCondition: String = s"f.${FOLDERS.CI_UID} = :ciUid"

  private def findAncestorByIdAndPermission(hasRoles: Boolean): String = findAncestorWithPermission(folderIdCondition, hasRoles)

  private val findAncestorById: String =
    s"""
       | WHERE $folderIdCondition
       """.stripMargin

  private def findAncestorByUidAndPermission(hasRoles: Boolean): String = findAncestorWithPermission(folderUidCondition, hasRoles)

  private val findAncestorByUid: String =
    s"""
       | WHERE $folderUidCondition
       """.stripMargin

  private val grouped: String =
    s"""
       | GROUP BY
       |  d.${VIEW.DESCENDANT_UID},
       |  p.${PATHS.ANCESTOR_UID},
       |  d.${VIEW.SECURITY_UID},
       |  d.${VIEW.TOKEN},
       |  d.${VIEW.FOLDER_PATH},
       |  d.${VIEW.FOLDER_ID},
       |  d.${VIEW.NAME},
       |  d.${VIEW.DEPTH}
       """.stripMargin

  private val topDownOrder: String =
    s"""
       | ORDER BY
       |   d.${VIEW.DEPTH} ASC,
       |   d.${VIEW.NAME} ASC,
       |   p.${PATHS.ANCESTOR_UID} ASC,
       |   d.${VIEW.DESCENDANT_UID} ASC
      """.stripMargin

  val toFolderData: RowMapper[FolderRow] = rowMapper { rs =>
    val uid = rs.getInt(VIEW.DESCENDANT_UID)
    if (uid == Root.uid) {
      Root
    } else {
      FolderEntry(
        uid = uid,
        token = Token.fromString(rs.getString(VIEW.TOKEN)).get,
        securityUid = rs.getInt(VIEW.SECURITY_UID),
        folderId = FolderId(rs.getString(VIEW.FOLDER_PATH)) / rs.getString(VIEW.FOLDER_ID),
        name = rs.getString(VIEW.NAME),
        parentUid = rs.getInt(PATHS.ANCESTOR_UID),
        depth = rs.getInt(VIEW.DEPTH)
      )
    }
  }

  def findByUid: String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorByUid ++
    grouped ++
    topDownOrder

  def findByUidHavingPermission(hasRoles: Boolean): String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorByUidAndPermission(hasRoles) ++
    grouped ++
    topDownOrder

  def findById: String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorById ++
    grouped ++
    topDownOrder

  def findByIdHavingPermission(hasRoles: Boolean): String = selectDescendants ++
    descendantsOf ++
    parentOf ++
    findAncestorByIdAndPermission(hasRoles) ++
    grouped ++
    topDownOrder
}
