package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{ROOT_FOLDER_ID, SEPARATOR, relativePathFrom}
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.FolderTreeUtils._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, PATHS, RELEASES, VIEW}
import com.xebialabs.xlrelease.repository.sql.persistence.Token._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow.{FolderEntry, Root}
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.utils.Tree.Node
import com.xebialabs.xlrelease.utils.{FolderId, Tree}
import org.springframework.jdbc.core.JdbcTemplate

import scala.collection.mutable
import scala.jdk.CollectionConverters._

class FolderPersistence(securablePersistence: SecurablePersistence)(val jdbcTemplate: JdbcTemplate, val dialect: Dialect)
  extends SqlRepository with PersistenceSupport {

  private val STMT_EXISTS: String = s"SELECT COUNT(*) FROM ${FOLDERS.TABLE} WHERE ${FOLDERS.FOLDER_ID} = :folderId"

  def exists(folderId: String): Boolean = {
    sqlQuery(STMT_EXISTS, params("folderId" -> FolderId(folderId).id), _.getInt(1) > 0).exists(e => e)
  }

  def create(name: String, givenId: String, parentIdOpt: Option[String]): Node[FolderRow] = {
    parentIdOpt
      .map(findById(_, depth = 0)) // find row in db, if parentId is given
      .getOrElse(Node(Root)) // assume parent = Root, if parentId is not given
      .fold(Tree.empty[FolderRow]) { parent => // if given and not found, return empty tree; otherwise:

        val folderId = parent.value.folderId / Ids.getName(givenId)
        val folderCiUid = securablePersistence.insert()
        // if parent is root, use own ciUid as securityUid, otherwise inherit from parent.
        val (parentUid, securityUid) = parent.value.fold(Root.uid -> folderCiUid)(p => p.uid -> p.securityUid)
        val (_, token) = insertFolder(name, folderId, securityUid, folderCiUid)
        insertPaths(folderCiUid, parentUid)
        Node(FolderEntry(folderCiUid, token, securityUid, folderId, name, parentUid, 0))
      }.toOption.getOrElse {
      throw new NotFoundException(s"parent folder with id ${parentIdOpt.getOrElse(ROOT_FOLDER_ID)} cannot be found")
    }
  }

  private val STMT_INSERT_FOLDER =
    s"""| INSERT INTO ${FOLDERS.TABLE}
        |   ( ${FOLDERS.CI_UID}
        |   , ${FOLDERS.SECURITY_UID}
        |   , ${FOLDERS.FOLDER_ID}
        |   , ${FOLDERS.FOLDER_PATH}
        |   , ${FOLDERS.NAME}
        |   , ${FOLDERS.TOKEN}
        |   )
        | VALUES
        |   ( :ciUid
        |   , :securityUid
        |   , :folderId
        |   , :folderPath
        |   , :name
        |   , :token
        |   )
      """.stripMargin

  private def insertFolder(name: String, folderId: FolderId, securityUid: CiUid, ciUid: CiUid): (CiUid, Token) = {
    val token = Token.fresh()
    sqlUpdate(STMT_INSERT_FOLDER, params(
      "ciUid" -> ciUid,
      "folderId" -> folderId.id,
      "folderPath" -> folderId.parent.path,
      "securityUid" -> securityUid,
      "name" -> name,
      "token" -> token.asString
    ), {
      case 1 => (ciUid, token)
      case _ => throw new IllegalStateException(s"Folder $folderId could not be created.")
    })
  }

  private def insertPaths(ciUid: CiUid, parentUid: CiUid): Int = {
    // add 1 row for each ancestor of :parentId, pointing to :ciUid with depth = :parentId.depth + 1
    // also adds 1 row for itself, from :ciUid to :ciUid at depth 0.
    val STMT_INSERT_PATHS =
    s"""|INSERT INTO ${PATHS.TABLE} (${PATHS.ANCESTOR_UID}, ${PATHS.DESCENDANT_UID}, ${PATHS.DEPTH})
        | SELECT p.${PATHS.ANCESTOR_UID}, $ciUid, p.${PATHS.DEPTH} + 1
        |   FROM ${PATHS.TABLE} p
        |   WHERE p.${PATHS.DESCENDANT_UID} = :parentUid
        | UNION ALL
        | SELECT f.${FOLDERS.CI_UID}, f.${FOLDERS.CI_UID}, 0
        |   FROM ${FOLDERS.TABLE} f WHERE f.${FOLDERS.CI_UID} = :ciUid""".stripMargin
    sqlUpdate(STMT_INSERT_PATHS, params(
      "ciUid" -> ciUid,
      "parentUid" -> parentUid // if previous part is FALSE, this value is not really used.
    ), identity)
  }

  private val STMT_GET_ALL_DESCENDANTS =
    s"""|SELECT p.${PATHS.DESCENDANT_UID}
        | FROM ${PATHS.TABLE} p
        | WHERE p.${PATHS.ANCESTOR_UID} = :folderUid""".stripMargin

  private val STMT_GET_ALL_ANCESTORS =
    s"""|SELECT p.${PATHS.ANCESTOR_UID}
        | FROM ${PATHS.TABLE} p
        | WHERE p.${PATHS.DESCENDANT_UID} = :folderUid
        | AND p.${PATHS.DEPTH} > 0""".stripMargin

  private val STMT_DELETE_EXTERNAL_PATHS =
    s"""|DELETE FROM ${PATHS.TABLE}
        | WHERE ${PATHS.DESCENDANT_UID} IN (:descendantUids)
        | AND ${PATHS.ANCESTOR_UID} IN (:ancestorUids)""".stripMargin

  private def deleteExternalPaths(ancestorUid: CiUid): Int = {
    val ancestorUids = sqlQuery(STMT_GET_ALL_ANCESTORS, params("folderUid" -> ancestorUid), _.getInt(PATHS.ANCESTOR_UID))
    val descendantUids = sqlQuery(STMT_GET_ALL_DESCENDANTS, params("folderUid" -> ancestorUid), _.getInt(PATHS.DESCENDANT_UID))
    sqlUpdate(STMT_DELETE_EXTERNAL_PATHS, params("ancestorUids" -> ancestorUids.asJava, "descendantUids" -> descendantUids.asJava), identity)
  }

  private val STMT_INSERT_SUBTREE_PATHS =
    s"""|INSERT INTO ${PATHS.TABLE} (${PATHS.ANCESTOR_UID}, ${PATHS.DESCENDANT_UID}, ${PATHS.DEPTH})
        | SELECT supertree.${PATHS.ANCESTOR_UID}, subtree.${PATHS.DESCENDANT_UID}, supertree.${PATHS.DEPTH} + subtree.${PATHS.DEPTH} + 1
        | FROM ${PATHS.TABLE} supertree
        | CROSS JOIN ${PATHS.TABLE} subtree
        | WHERE subtree.${PATHS.ANCESTOR_UID} = :ancestorUid AND supertree.${PATHS.DESCENDANT_UID} = :newParentUid""".stripMargin

  private def insertSubTreePaths(ancestorUid: CiUid, newParentUid: CiUid): Int = {
    sqlUpdate(STMT_INSERT_SUBTREE_PATHS, params("ancestorUid" -> ancestorUid, "newParentUid" -> newParentUid), identity)
  }

  private val STMT_UPDATE_SECURITY_UID =
    s"""|UPDATE ${FOLDERS.TABLE}
        | SET ${FOLDERS.SECURITY_UID} = :newSecurityUid
        | WHERE ${FOLDERS.SECURITY_UID} = :oldSecurityUid
        |   AND ${FOLDERS.CI_UID} IN (:descendantUids)""".stripMargin

  private def updateSubtreeSecurityUid(oldSecurityUid: CiUid, newSecurityUid: CiUid, newParentUid: CiUid): Unit = {
    val descendantUids = sqlQuery(STMT_GET_ALL_DESCENDANTS, params("folderUid" -> newParentUid), _.getInt(PATHS.DESCENDANT_UID))
    sqlUpdate(STMT_UPDATE_SECURITY_UID, params(
      "descendantUids" -> descendantUids.asJava,
      "oldSecurityUid" -> oldSecurityUid,
      "newSecurityUid" -> newSecurityUid
    ), _ => ())
  }

  private val STMT_UPDATE_FOLDER_PATH =
    s"UPDATE ${FOLDERS.TABLE} SET ${FOLDERS.FOLDER_PATH} = :folderPath WHERE ${FOLDERS.CI_UID} = :ciUid"

  private def updateFolderId(ciUid: CiUid, newFolderId: FolderId): Unit = {
    sqlUpdate(STMT_UPDATE_FOLDER_PATH, params("ciUid" -> ciUid, "folderPath" -> newFolderId.parent.path), _ => ())
  }

  private def moveSubTree(subTree: Tree[FolderRow]): Unit = {
    subTree.fold(()) { case Node(value, children) =>
      updateFolderId(value.uid, value.folderId)
      children.foreach(moveSubTree(_))
    }
  }

  private def computeNewSecurityUid(entry: FolderEntry, ancestor: FolderRow, newAncestorParent: FolderRow) = {
    if (ancestor.inheritsSecurity && entry.inheritsSecurity && entry.securityUid == ancestor.securityUid) {
      newAncestorParent.securityUid
    } else {
      entry.securityUid
    }
  }

  def move(folderId: String, newParentId: String, token: Option[Token] = None): Node[FolderRow] = {
    val oldParentId = Ids.getParentId(folderId)
    val subTree = findById(folderId)
    val newParentTree = findById(newParentId, depth = 1)
    if (newParentTree.children.exists(_.value.name == subTree.value.name)) {
      throw new ItemConflictException(s"Folder with title '${subTree.value.name}' already exists under folder by ID '${newParentTree.value.folderId.absolute}'")
    } else {
      deleteExternalPaths(subTree.value.uid)
      insertSubTreePaths(subTree.value.uid, newParentTree.value.uid)

      val updatedSubTree = subTree.map(node =>
        node.fold[FolderRow](Root)(subNode => {
          val newFolderId = newParentTree.value.folderId / relativePathFrom(oldParentId, subNode.folderId.absolute)
          val newSecurityUid = computeNewSecurityUid(subNode, subTree.value, newParentTree.value)
          subNode.copy(folderId = newFolderId, securityUid = newSecurityUid)
        })
      )

      moveSubTree(updatedSubTree)

      if (subTree.value.inheritsSecurity) {
        updateSubtreeSecurityUid(subTree.value.securityUid, newParentTree.value.securityUid, subTree.value.uid)
      }

      updatedSubTree
    }
  }

  private val STMT_RENAME: String =
    s"""|UPDATE ${FOLDERS.TABLE}
        | SET ${FOLDERS.NAME} = :newTitle
        | WHERE ${FOLDERS.FOLDER_ID} = :folderId""".stripMargin

  def rename(folderId: String, newTitle: String, token: Option[Token] = None): Folder = {

    val parentFolder = findById(Ids.getParentId(folderId), depth = 1).toFolder
    val parentChildren = parentFolder.getChildren.asScala
    if (parentChildren.exists(f => f.getId != folderId && f.getTitle == newTitle)) {
      throw new ItemConflictException(s"Folder with title '$newTitle' already exists under folder by ID '${parentFolder.getId}'")
    }
    val folder: Folder = parentChildren.find(_.getId == folderId).get
    sqlUpdate(STMT_RENAME, params("folderId" -> FolderId(folder.getId).id, "newTitle" -> newTitle), numRows => {
      if (numRows != 1) {
        throw new IllegalStateException(s"Folder ${folder.getId} could not be renamed to $newTitle")
      }
      folder.setTitle(newTitle)
      folder
    })
  }

  private val STMT_GET_UID = s"SELECT ${FOLDERS.CI_UID} FROM ${FOLDERS.TABLE} WHERE ${FOLDERS.FOLDER_ID} = :folderId"

  def getUid(folderId: String): CiUid = {
    sqlQuery(STMT_GET_UID, params("folderId" -> FolderId(folderId).id), _.getInt(FOLDERS.CI_UID)).headOption.getOrElse {
      throw new NotFoundException(s"Folder $folderId not found")
    }
  }

  def getUidsByIds(folderIds: Iterable[CiId]): Map[CiId, CiUid] = {
    if (folderIds.nonEmpty) {
      val stmt =
        s"""|SELECT ${FOLDERS.CI_UID}, ${FOLDERS.FOLDER_ID}
            |FROM ${FOLDERS.TABLE}
            |WHERE ${FOLDERS.FOLDER_ID} IN (:folderIds)
       """.stripMargin
      sqlQuery(stmt,
        params("folderIds" -> folderIds.map(FolderId(_).id).asJava),
        rs => rs.getString(FOLDERS.FOLDER_ID) -> CiUid(rs.getInt(FOLDERS.CI_UID))
      ).toMap
    } else {
      Map.empty
    }
  }

  private val STMT_FIND_BY_UID = FindDescendantsQueryBuilder.findByUid

  def findByUid(ciUid: CiUid, depth: Int = Int.MaxValue): Node[FolderRow] = {
    buildTree(
      sqlQuery(STMT_FIND_BY_UID, params("ciUid" -> ciUid, "depth" -> depth), FindDescendantsQueryBuilder.toFolderData)
    ).toOption.getOrElse {
      throw new NotFoundException(s"Folder with uid $ciUid not found")
    }
  }

  def findByUidHavingPermission(ciUid: CiUid,
                                permission: Permission,
                                principals: Iterable[String],
                                roles: Iterable[String],
                                depth: Int = Int.MaxValue): Node[FolderRow] = {
    buildTree(
      sqlQuery(FindDescendantsQueryBuilder.findByUidHavingPermission(roles.nonEmpty),
        params(
          "ciUid" -> ciUid,
          "depth" -> depth,
          "principalNames" -> principals.map(_.toLowerCase).asJava,
          "permissionName" -> permission.getPermissionName,
          "roles" -> roles.asJava
        ), FindDescendantsQueryBuilder.toFolderData)
    ).toOption.getOrElse {
      throw new NotFoundException(s"Folder with uid $ciUid not found")
    }
  }

  private val STMT_FIND_BY_ID = FindDescendantsQueryBuilder.findById

  def findById(folderId: String, depth: Int = Int.MaxValue): Node[FolderRow] = {
    buildTree(
      sqlQuery(STMT_FIND_BY_ID, params("folderId" -> FolderId(folderId).id, "depth" -> depth), FindDescendantsQueryBuilder.toFolderData)
    ).toOption.getOrElse {
      throw new NotFoundException(s"Folder $folderId not found")
    }
  }

  def findDescendantsById(folderId: String, depth: Int = Int.MaxValue): mutable.Buffer[FolderRow] = {
    sqlQuery(STMT_FIND_BY_ID, params("folderId" -> FolderId(folderId).id, "depth" -> depth), FindDescendantsQueryBuilder.toFolderData)
  }

  def findByIdHavingPermission(folderId: String,
                               permission: Permission,
                               principals: Iterable[String],
                               roles: Iterable[String],
                               depth: Int = Int.MaxValue): Node[FolderRow] = {
    buildTree(
      findDescendantsByIdHavingPermission(folderId, permission, principals, roles, depth)
    ).toOption.getOrElse {
      throw new NotFoundException(s"Folder $folderId not found")
    }
  }

  def findDescendantsByIdHavingPermission(folderId: String,
                                          permission: Permission,
                                          principals: Iterable[String],
                                          roles: Iterable[String],
                                          depth: Int = Int.MaxValue): mutable.Buffer[FolderRow] = {
    sqlQuery(FindDescendantsQueryBuilder.findByIdHavingPermission(roles.nonEmpty),
      params(
        "folderId" -> FolderId(folderId).id,
        "depth" -> depth,
        "principalNames" -> principals.map(_.toLowerCase).asJava,
        "permissionName" -> permission.getPermissionName,
        "roles" -> roles.asJava
      ),
      FindDescendantsQueryBuilder.toFolderData)
  }

  private val STMT_GET_EFFECTIVE_SECURED_CI =
    s"""|SELECT
        |   s.${FOLDERS.FOLDER_PATH},
        |   s.${FOLDERS.FOLDER_ID},
        |   s.${FOLDERS.CI_UID}
        | FROM ${FOLDERS.TABLE} s
        | JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.SECURITY_UID} = s.${FOLDERS.CI_UID}
        | WHERE f.${FOLDERS.FOLDER_ID} = :folderId
     """.stripMargin

  def getEffectiveSecuredCi(folderId: String): SecuredCi = {
    sqlQuery(STMT_GET_EFFECTIVE_SECURED_CI,
      params("folderId" -> FolderId(folderId).id),
      rs => {
        val containerId = FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)
        new SecuredCi(containerId.absolute, rs.getString(FOLDERS.CI_UID))
      }
    ).headOption.getOrElse {
      throw new NotFoundException(s"Folder '$folderId' does not exist in the repository")
    }
  }

  private val STMT_GET_SECURED_CI =
    s"""|SELECT
        |   f.${FOLDERS.FOLDER_PATH},
        |   f.${FOLDERS.FOLDER_ID},
        |   f.${FOLDERS.CI_UID}
        | FROM ${FOLDERS.TABLE} f
        | WHERE f.${FOLDERS.FOLDER_ID} = :folderId""".stripMargin

  def getSecuredCi(folderId: String): SecuredCi = {
    sqlQuery(STMT_GET_SECURED_CI,
      params("folderId" -> FolderId(folderId).id),
      rs => {
        val containerId = FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)
        new SecuredCi(containerId.absolute, rs.getString(FOLDERS.CI_UID))
      }
    ).headOption.getOrElse {
      throw new NotFoundException(s"Folder $folderId not found")
    }
  }

  private val STMT_GET_ALL_INHERITING_DESCENDANTS =
    s"""|SELECT v.${VIEW.DESCENDANT_UID}
        | FROM ${FOLDERS.TABLE} f
        | JOIN ${VIEW.TABLE} v ON ( v.${VIEW.ANCESTOR_UID} = f.${FOLDERS.CI_UID} AND v.${VIEW.SECURITY_UID} = f.${FOLDERS.SECURITY_UID} )
        | WHERE f.${FOLDERS.CI_UID} = :parentUid""".stripMargin

  private val STMT_SET_SECURITY_ID =
    s"""|UPDATE ${FOLDERS.TABLE}
        | SET
        |   ${FOLDERS.SECURITY_UID} = :securityUid
        | WHERE
        |   ${FOLDERS.CI_UID} IN (:descendantUids)""".stripMargin
  // additional subquery was added to fix issue with mysql & maria: same table in update query and as a datasource

  private def updateSecurityUid(folderData: FolderRow, securityUid: CiUid) = {
    val descendantUids = sqlQuery(STMT_GET_ALL_INHERITING_DESCENDANTS, params("parentUid" -> folderData.uid), _.getInt(VIEW.DESCENDANT_UID))
    sqlUpdate(STMT_SET_SECURITY_ID, params("descendantUids" -> descendantUids.asJava, "securityUid" -> securityUid), _ => ())
    folderData match {
      case f: FolderEntry => f.copy(securityUid = securityUid)
      case Root => Root
    }
  }

  def setAsEffectiveSecuredCi(folderId: String): FolderRow = {
    val folderData = findById(folderId, depth = 0).value
    val securityUid = folderData.uid
    updateSecurityUid(folderData, securityUid)
  }

  def inheritEffectiveSecuredCi(folderId: String): FolderRow = {
    val folderData = findById(folderId, depth = 0).value
    val securityUid = findById(Ids.getParentId(folderId), 0).value.securityUid
    updateSecurityUid(folderData, securityUid)
  }

  private val STMT_NAME_EXISTS_IN_FOLDER: String =
    s"""|SELECT
        |  d.${VIEW.DESCENDANT_UID},
        |  d.${VIEW.TOKEN},
        |  d.${VIEW.SECURITY_UID},
        |  d.${VIEW.FOLDER_ID},
        |  d.${VIEW.FOLDER_PATH},
        |  d.${VIEW.ANCESTOR_UID}
        | FROM ${VIEW.TABLE} d
        | JOIN ${FOLDERS.TABLE} f ON ( f.${FOLDERS.CI_UID} = d.${VIEW.ANCESTOR_UID} AND d.${VIEW.DEPTH} = 1 )
        | WHERE f.${FOLDERS.FOLDER_ID} = :parentId
        |   AND d.${VIEW.NAME} = :name""".stripMargin

  def findSubFolderByTitle(name: String, parentId: String): Option[FolderRow] = {
    sqlQuery(
      STMT_NAME_EXISTS_IN_FOLDER,
      params("parentId" -> FolderId(parentId).id, "name" -> name),
      rs =>
        FolderEntry(
          CiUid(rs.getInt(VIEW.DESCENDANT_UID)),
          Token.fromString(rs.getString(VIEW.TOKEN)).get,
          CiUid(rs.getInt(VIEW.SECURITY_UID)),
          FolderId(rs.getString(VIEW.FOLDER_PATH)) / rs.getString(VIEW.FOLDER_ID),
          name,
          CiUid(rs.getInt(VIEW.ANCESTOR_UID)),
          1
        )
    ).headOption
  }

  private val STMT_FIND_ALL_RELEASE_UIDS: String =
    s"""|SELECT
        |  r.${RELEASES.CI_UID},
        |  f.${FOLDERS.FOLDER_ID},
        |  r.${RELEASES.RELEASE_ID}
        | FROM ${PATHS.TABLE} p
        | JOIN ${RELEASES.TABLE} r ON p.${PATHS.DESCENDANT_UID} = r.${RELEASES.FOLDER_CI_UID}
        | JOIN ${FOLDERS.TABLE} f ON r.${RELEASES.FOLDER_CI_UID} = f.${FOLDERS.CI_UID}
        | WHERE p.${PATHS.ANCESTOR_UID} = :ciUid""".stripMargin

  def deleteReleases(ciUid: CiUid, deleteRelease: (Int, String) => Unit): Unit = {
    val releaseTuple = sqlQuery(STMT_FIND_ALL_RELEASE_UIDS, params("ciUid" -> ciUid),
      rs => rs.getInt(RELEASES.CI_UID) -> (rs.getString(FOLDERS.FOLDER_ID) + SEPARATOR + rs.getString(RELEASES.RELEASE_ID))
    )
    releaseTuple.foreach(deleteRelease.tupled)
  }

  def deleteByUid(ciUid: CiUid): Unit = {
    if (ciUid == Root.uid) {
      throw new IllegalArgumentException("Cannot delete root folder!")
    } else {
      if (deleteFolderRow(ciUid)) {
        securablePersistence.deleteCi(ciUid)
      } else {
        throw new NotFoundException(s"Folder with UID $ciUid not found")
      }
    }
  }

  private val STMT_DELETE_FOLDER =
    s"""DELETE FROM ${FOLDERS.TABLE} WHERE ${FOLDERS.CI_UID} = :ciUid""".stripMargin

  private val STMT_DELETE_FOLDER_PATHS =
    s"""DELETE FROM ${PATHS.TABLE} WHERE ${PATHS.ANCESTOR_UID} = :ciUid OR ${PATHS.DESCENDANT_UID} = :ciUid""".stripMargin

  private def deleteFolderRow(ciUid: CiUid): Boolean = {
    sqlUpdate(STMT_DELETE_FOLDER_PATHS, params("ciUid" -> ciUid), {
      case 0 => false
      case _ => sqlUpdate(STMT_DELETE_FOLDER, params("ciUid" -> ciUid), {
        case 0 => false
        case _ => true
      })
    })
  }

  private val STMT_FOLDER_PATH_SEGMENTS =
    s"""
       |SELECT AF.NAME
       |FROM
       |  XLR_FOLDERS F
       |  JOIN XLR_FOLDER_PATHS P ON P.DESCENDANT_UID = F.CI_UID
       |  JOIN XLR_FOLDERS AF ON AF.CI_UID = P.ANCESTOR_UID
       |WHERE
       |  F.FOLDER_ID = :folderId
       |ORDER BY P.DEPTH DESC
     """.stripMargin

  def getFolderPathSegments(folderId: String): Seq[String] = {
    sqlQuery(STMT_FOLDER_PATH_SEGMENTS, params("folderId" -> FolderId(folderId).id), _.getString(1)).toSeq
  }

  private val STMT_FOLDER_IDS_BY_SECURITY_UID =
    s"SELECT COUNT(*) FROM ${FOLDERS.TABLE} WHERE ${FOLDERS.FOLDER_ID} = :folderId AND ${FOLDERS.CI_UID} <> ${FOLDERS.SECURITY_UID}"

  def isInherited(folderId: String): Boolean = {
    sqlQuery(STMT_FOLDER_IDS_BY_SECURITY_UID, params("folderId" -> FolderId(folderId).id), _.getInt(1) > 0).head
  }

  private val STMT_CHILDREN_OF_A_FOLDER =
    s"SELECT MAX( ${PATHS.DEPTH}) FROM ${PATHS.TABLE} WHERE ${PATHS.ANCESTOR_UID} IN " +
      s"(SELECT ${FOLDERS.CI_UID} FROM ${FOLDERS.TABLE} WHERE ${FOLDERS.FOLDER_ID} = :folderId)"

  def findNumberOfChildrenForAFolder(folderId: String): Int = {
    sqlQuery(STMT_CHILDREN_OF_A_FOLDER, params("folderId" -> folderId), _.getInt(1)).head
  }

}
