package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.METADATA
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.jdbc.core.JdbcTemplate

@IsTransactional
class MetadataPersistence @Autowired()(implicit val jdbcTemplate: JdbcTemplate,
                                       implicit val dialect: Dialect) extends PersistenceSupport {

  private def params(map: (String, _ <: Any)*): Map[String, Any] = map.toMap[String, Any]

  def isTableExists: Boolean = {
    val tableName = Schema.METADATA.TABLE
    val tables = jdbcTemplate.getDataSource.getConnection.getMetaData.getTables("", "APP", tableName, Array("TABLE"))
    try {
      if (!tables.next()) {
        false
      } else {
        tables.getString("TABLE_NAME") == tableName
      }
    } finally {
      tables.close()
    }
  }

  private val STMT_INSERT_METADATA =
    s"""INSERT INTO ${METADATA.TABLE} ( ${METADATA.NAME}, ${METADATA.ENTRY} )
       |VALUES ( :name, :entry )""".stripMargin

  def insert(name: String, entry: String): Unit = {
    sqlUpdate(
      STMT_INSERT_METADATA,
      params(
        "name" -> name,
        "entry" -> entry
      ), _ => ())
  }

  private val STMT_UPDATE_METADATA =
    s"""UPDATE ${METADATA.TABLE}
       |SET ${METADATA.ENTRY} = :entry
       |where ${METADATA.NAME} = :name""".stripMargin

  def update(name: String, entry: String): Unit = {
    sqlUpdate(
      STMT_UPDATE_METADATA,
      params(
        "name" -> name,
        "entry" -> entry
      ), _ => ())
  }

  private val STMT_DELETE_METADATA_BY_NAME =
    s"""|DELETE FROM ${METADATA.TABLE}
        | WHERE ${METADATA.NAME} = :name""".stripMargin

  def delete(name: String): Unit = {
    sqlExec(
      STMT_DELETE_METADATA_BY_NAME,
      params(
        "name" -> name
      ), _.execute())
  }

  private val STMT_EXISTS_BY_NAME = s"SELECT COUNT(${METADATA.NAME}) FROM ${METADATA.TABLE} WHERE ${METADATA.NAME} = :name"

  def exists(name: String): Boolean = {
    sqlQuery(
      STMT_EXISTS_BY_NAME,
      params(
        "name" -> name
      ), _.getInt(1) > 0).exists(e => e)
  }

  private val STMT_GET_ENTRY_BY_NAME = s"SELECT ${METADATA.ENTRY} FROM ${METADATA.TABLE} WHERE ${METADATA.NAME} IN (:name)"

  def getEntry(name: String): String = {
    findOne(sqlQuery(STMT_GET_ENTRY_BY_NAME,
      params("name" -> name),
      rs => rs.getString(METADATA.ENTRY))).get
  }

}
