package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.PermissionEnforcer
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.{ParallelGroup, SequentialGroup}
import com.xebialabs.xlrelease.repository.sql.persistence.TasksSqlBuilder
import com.xebialabs.xlrelease.views.TasksFilters
import org.springframework.util.StringUtils._

import scala.jdk.CollectionConverters._

object SqlTasksFilterSupport extends SqlFilterSupport[TasksFilters, TasksSqlBuilder] {

  override def sqlBuilderByFilters(filters: TasksFilters, principals: Iterable[String], roleIds: Iterable[String], enforcePermission: Boolean = true)
                                  (implicit permissionEnforcer: PermissionEnforcer, sqlDialect: Dialect): TasksSqlBuilder = {

    val builder = new TasksSqlBuilder()
    builder.selectTaskId()
    builder.withOneOfReleaseStatuses(Seq(ReleaseStatus.ACTIVE_STATUSES: _*))
    builder.withOneOfTaskStatuses(getTaskStatuses(filters).toSeq)
    builder.withTaskTypeNot(Type.valueOf(classOf[ParallelGroup]))
    builder.withTaskTypeNot(Type.valueOf(classOf[SequentialGroup]))
    builder.withFilter(filters.getFilter)
    builder.withDates(filters.getFrom, filters.getTo)
    builder.withAnyOfTags(filters.getTags.asScala.toSet)
    if (hasText(filters.getFolderId)) {
      builder.withFolderIds(Set(filters.getFolderId))
    }
    builder.orderByTaskId()

    if (enforcePermission) {
      if (filters.isAssignedToAnybody) {
        builder.withAssignedToAnybody(principals, roleIds, permissionEnforcer.isCurrentUserAdmin)
      } else if (filters.isAssignedToMyTeams) {
        builder.withAssignedToMeOrMyTeams(principals, roleIds)
      } else {
        builder.withAssignedToMe(principals)
      }
    }

    builder
  }

  private def getTaskStatuses(filters: TasksFilters) = if (filters.getStatuses.isEmpty) {
    TaskStatus.values().toSeq.filter(ts => !ts.isDone && !ts.isDoneInAdvance)
  } else {
    filters.getStatuses.asScala
  }
}
