package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlplatform.utils.ResourceManagement
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{RELEASES, RELEASES_DATA}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import grizzled.slf4j.Logging
import org.codehaus.jettison.json.JSONObject
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component

import java.sql.ResultSet
import scala.jdk.CollectionConverters._


@Component
class XLRelease980OriginTemplateIdUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                       @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect
                                                      )
  extends Upgrade
    with PersistenceSupport
    with Logging
    with LimitOffset {

  import XLRelease980OriginTemplateIdUpgrade._

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.8.0#3")

  private val SQL_IN_BATCH_SIZE: Int = 500

  override def doUpgrade(): Boolean = {
    logger.info("Starting origin template ID extraction upgrade")

    LazyList.iterate(getReleasesByTemplateId(0, SQL_IN_BATCH_SIZE))(result => getReleasesByTemplateId(result.page + 1, SQL_IN_BATCH_SIZE))
      .takeWhile(_.count > 0)
      .foreach { result =>
        logger.info(s"Upgrading ${result.count} releases from ${result.groupedReleases.size} templates...")

        result.groupedReleases.foreach { item =>
          namedTemplate.update(
            STMT_UPDATE_RELEASE,
            Map[String, Any](ORIGIN_TEMPLATE_ID -> item.originTemplateId, "ciUids" -> item.releaseCiUids.asJava).asJava
          )
        }
      }

    logger.info("Finished origin template ID extraction upgrade")
    true
  }

  private def getReleasesByTemplateId(page: Int, numberByPage: Int): ReleaseCiUidsByTemplateIdResult = {
    val query = addLimitAndOffset(STMT_SELECT_RELEASES, numberByPage, page * numberByPage)
    val rows = sqlQuery(query, params(), originTemplateDataRowMapper)
    val grouped = rows
      .filter(_.originTemplateId != null)
      .groupBy(_.originTemplateId)
      .map { case (originTemplateId, ciUids) => ReleaseCiUidsByTemplateId(originTemplateId, ciUids.flatMap(_.releaseCiUids).toSeq) }
      .toSeq

    ReleaseCiUidsByTemplateIdResult(page, rows.size, grouped)
  }

  private val originTemplateDataRowMapper: RowMapper[ReleaseCiUidsByTemplateId] = (rs: ResultSet, _: Int) => {
    val releaseData = ResourceManagement.using(rs.getBinaryStream(RELEASES_DATA.CONTENT))(decompress)

    val releaseJson = new JSONObject(releaseData)
    val originTemplateId = if (releaseJson.has(ORIGIN_TEMPLATE_ID)) {
      Ids.getName(releaseJson.getString(ORIGIN_TEMPLATE_ID))
    } else {
      null
    }
    val releaseCiUid = rs.getInt(RELEASES.CI_UID)

    ReleaseCiUidsByTemplateId(originTemplateId, Seq(releaseCiUid))
  }

}

object XLRelease980OriginTemplateIdUpgrade {
  val ORIGIN_TEMPLATE_ID = "originTemplateId"

  val STMT_SELECT_RELEASES: String =
    s"""|SELECT
        |  rel.${RELEASES.CI_UID},
        |  d.${RELEASES_DATA.CONTENT}
        | FROM ${RELEASES.TABLE} rel
        | INNER JOIN ${RELEASES_DATA.TABLE} d ON rel.${RELEASES.CI_UID} = d.${RELEASES_DATA.CI_UID}
        | WHERE rel.${RELEASES.STATUS} != '${ReleaseStatus.TEMPLATE.value()}'
        | ORDER BY rel.${RELEASES.CI_UID}""".stripMargin

  val STMT_UPDATE_RELEASE: String =
    s"""|UPDATE ${RELEASES.TABLE}
        | SET
        |   ${RELEASES.ORIGIN_TEMPLATE_ID} = :originTemplateId
        | WHERE ${RELEASES.CI_UID} IN (:ciUids)""".stripMargin

  case class ReleaseCiUidsByTemplateIdResult(page: Int, count: Int, groupedReleases: Seq[ReleaseCiUidsByTemplateId])

  case class ReleaseCiUidsByTemplateId(originTemplateId: String, releaseCiUids: Seq[Int])

}

