package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.PlanItem
import com.xebialabs.xlrelease.repository.IdMatchers.{PhaseId, ReleaseId, TaskId}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.ReleaseJsonParser.PhaseLevel
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{ReleasePersistence, TaskPersistence}
import com.xebialabs.xlrelease.repository.{IdMatchers, Ids, PlanItemRepository, ReleaseJsonParser}
import grizzled.slf4j.Logging

import scala.util.Try

@IsTransactional
class SqlPlanItemRepository(releasePersistence: ReleasePersistence,
                            taskPersistence: TaskPersistence,
                            val repositoryAdapter: SqlRepositoryAdapter)
  extends PlanItemRepository with DeserializationSupport with Logging {

  @Timed
  @IsReadOnly
  override def findById[T <: PlanItem](ciId: CiId): T = {
    import IdMatchers._

    ciId.normalized match {
      case ReleaseId(id) => findRelease(id).asInstanceOf[T]
      case PhaseId(id) => findRelease(releaseIdFrom(id)).getPhase(id).asInstanceOf[T]
      case TaskId(id) => findRelease(releaseIdFrom(id)).getTask(id).asInstanceOf[T]
      case id => throw new NotFoundException(s"Item [$id] not found")
    }
  }

  private def findRelease[Release](id: CiId) = {
    val releaseId = releaseIdFrom(id.normalized)
    releasePersistence.findByReleaseId(releaseId)
      .map(deserializeRelease(_))
      .getOrElse(throw new NotFoundException(s"Release [$id] not found"))
  }

  override def exists(id: String): Boolean = {
    Try(Ids.releaseIdFrom(id).normalized).map { releaseId =>
      id match {
        case ReleaseId(_) => releasePersistence.existsRelease(releaseId)
        case PhaseId(_) =>
          releasePersistence.getReleaseJson(releaseId) match {
            case Some(releaseJson) => Try(ReleaseJsonParser.parse(releaseJson, PhaseLevel).getPhase(id)).isSuccess
            case None => false
          }
        case TaskId(_) => taskPersistence.exists(id)
        case _ =>
          val ex = new IllegalArgumentException(s"$id is not a PlanItem ID")
          logger.error(s"$id is not a PlanItem ID", ex)
          false
      }
    }.getOrElse(false)
  }
}


