package com.xebialabs.xlrelease.repository.sql

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.api.v1.filter.CategoryFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.Category
import com.xebialabs.xlrelease.repository.CategoryRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.CATEGORIES
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{RichIntAsBoolean, RichJBooleanAsInt, params}
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.repository.sql.query.CategoryQueryBuilder
import com.xebialabs.xlrelease.repository.sql.query.CategoryQueryBuilder.STMT_CATEGORY_SELECT
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

@IsTransactional
class SqlCategoryRepository(implicit val dialect: Dialect,
                            implicit val jdbcTemplate: JdbcTemplate)
  extends CategoryRepository
    with CategoryMapper
    with PersistenceSupport
    with Utils
    with Logging {

  private val STMT_INSERT_CATEGORY =
    s"""INSERT INTO ${CATEGORIES.TABLE} (
       |  ${CATEGORIES.CATEGORY},
       |  ${CATEGORIES.ACTIVE}
       | ) VALUES (
       |  :${CATEGORIES.CATEGORY},
       |  :${CATEGORIES.ACTIVE}
       | )
       |""".stripMargin

  @Timed
  override def create(category: Category): Category = {
    logger.trace(s"creating category $category")
    val params = Map(
      CATEGORIES.CATEGORY -> category.getTitle,
      CATEGORIES.ACTIVE -> category.getActive.asInteger
    )

    val ciUid = sqlInsert(pkName(CATEGORIES.CI_UID), STMT_INSERT_CATEGORY, params)

    category.setCiUid(ciUid)
    category
  }

  private val STMT_UPDATE_CATEGORY =
    s"""UPDATE ${CATEGORIES.TABLE}
       |  SET
       |    ${CATEGORIES.CATEGORY} = :${CATEGORIES.CATEGORY},
       |    ${CATEGORIES.ACTIVE} = :${CATEGORIES.ACTIVE}
       |  WHERE ${CATEGORIES.CI_UID} = :${CATEGORIES.CI_UID}
       |""".stripMargin

  @Timed
  override def update(category: Category): Category = {
    logger.trace(s"updating category $category")
    val params = Map(
      CATEGORIES.CATEGORY -> category.getTitle,
      CATEGORIES.ACTIVE -> category.getActive.asInteger,
      CATEGORIES.CI_UID -> category.getCiUid
    )

    sqlUpdate(STMT_UPDATE_CATEGORY, params, _ => ())
    category
  }

  private val STMT_DELETE_CATEGORY = {
    s"DELETE FROM ${CATEGORIES.TABLE} WHERE ${CATEGORIES.CI_UID} = :${CATEGORIES.CI_UID}"
  }

  @Timed
  override def delete(id: Int): Unit = {
    logger.trace(s"deleting category with id[$id]")
    sqlExec(STMT_DELETE_CATEGORY, params(CATEGORIES.CI_UID -> id), _.execute())
  }

  private val STMT_FIND_CATEGORY_BY_ID =
    s"""
       |$STMT_CATEGORY_SELECT
       | WHERE ${CATEGORIES.CI_UID} = ?
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(System.lineSeparator)

  @Timed
  @IsReadOnly
  override def findByCiUid(id: Int): Option[Category] = {
    findOptional(_.queryForObject(STMT_FIND_CATEGORY_BY_ID, categoryMapper, id))
  }

  private val STMT_FIND_CATEGORY_BY_TITLE =
    s"""
       |$STMT_CATEGORY_SELECT
       | WHERE LOWER(${CATEGORIES.CATEGORY}) = ?
       |""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(System.lineSeparator)

  @Timed
  @IsReadOnly
  override def findByTitle(title: String): Option[Category] = {
    findOptional(_.queryForObject(STMT_FIND_CATEGORY_BY_TITLE, categoryMapper, title.toLowerCase))
  }

  @Timed
  @IsReadOnly
  override def findBy(categoryFilters: CategoryFilters, pageable: Pageable): Page[Category] = {
    CategoryQueryBuilder(dialect, namedTemplate)
      .from(categoryFilters)
      .withPageable(pageable)
      .build()
      .execute()
  }

}

trait CategoryMapper {
  val categoryMapper: RowMapper[Category] = (rs, _) => {
    val category: Category = new Category()
    category.setCiUid(rs.getInt(CATEGORIES.CI_UID))
    category.setTitle(rs.getString(CATEGORIES.CATEGORY))
    category.setActive(rs.getInt(CATEGORIES.ACTIVE).asBoolean)
    category
  }
}

