package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, LazyConfigurationItem}
import com.xebialabs.deployit.repository.RepositoryAdapter
import com.xebialabs.deployit.service.validation.Validator
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.facet.{Facet, TaskReportingRecord}
import com.xebialabs.xlrelease.domain.utils.syntax._
import com.xebialabs.xlrelease.domain.variables.{ValueProviderConfiguration, Variable}
import com.xebialabs.xlrelease.features.ReleaseLoadingFeature
import com.xebialabs.xlrelease.json.JsonUtils.{ReleaseJsonOps, TaskFacetJsonOps}
import com.xebialabs.xlrelease.repository.IdMatchers._
import com.xebialabs.xlrelease.repository.Ids.getFolderlessId
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence.ConfigurationRow
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.sql.persistence.data.{DependencyRow, ReleaseRow}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, DependencyPersistence, FacetPersistence, ReleasePersistence, TaskPersistence}
import com.xebialabs.xlrelease.repository.sql.proxy.{CiReferenceTargetTypeResolver, ProxyBasedResolverRepository}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.serialization.json.xltype.{CiJson2Reader, XlrPasswordEncryptingCiConverter}
import com.xebialabs.xlrelease.service.{ArchivedReleaseReader, NonDecryptingPasswordEncrypter}
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xltype.serialization.{CiReference, SerializationException}
import grizzled.slf4j.Logging

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._
import scala.util.{Success, Try}

// this is internal class - do not use it outside SqlRepositoryAdapter
// scalastyle:off number.of.methods
private[sql] case class SqlRepositoryAdapterWithReadsCache(releasePersistence: ReleasePersistence,
                                                           dependencyPersistence: DependencyPersistence,
                                                           configurationPersistence: ConfigurationReferencePersistence,
                                                           facetPersistence: FacetPersistence,
                                                           taskPersistence: TaskPersistence,
                                                           validator: Validator,
                                                           encrypter: NonDecryptingPasswordEncrypter,
                                                           referenceTargetTypeResolver: CiReferenceTargetTypeResolver,
                                                           archivedReleases: ArchivedReleases,
                                                           resolveOptions: ResolveOptions,
                                                           resolveContextCis: mutable.Map[CiId, ConfigurationItem] = mutable.Map.empty
                                                          )
  extends RepositoryAdapter(null, validator, null, null)
    with ProxyBasedResolverRepository
    with ArchivedReleaseReader
    with Logging {

  override def repositoryAdapter: RepositoryAdapter = this

  val alreadyLoadedReleases: mutable.HashSet[CiId] = mutable.HashSet.empty

  val traceLog = new MessageLogger

  def createCiConverter(): XlrPasswordEncryptingCiConverter = {
    val cisInContext = resolveContextCis.values.toSeq.asJava
    new XlrPasswordEncryptingCiConverter(encrypter, cisInContext, resolveOptions)
  }

  override def read[T <: ConfigurationItem](ciId: CiId): T = {
    readInternal(ciId)
  }

  override def read[T <: ConfigurationItem](ids: util.List[CiId], depth: Integer): util.List[T] = {
    ids.asScala.map(readInternal[T]).filter(_ != null).asJava
  }

  override def doResolve(id: String, ciReference: CiReference): ConfigurationItem = {
    val res = resolveContextCis.get(id)
    res.getOrElse {
      val resolved = super.doResolve(id, ciReference)
      if (resolved != null) cache(resolved)
      resolved
    }
  }

  override def read[T <: ConfigurationItem](id: CiId, target: ConfigurationItem): T = {
    cacheTargetConfigurationItem(id, target)
    val result = if (Ids.isInRelease(id) && alreadyLoadedReleases.contains(Ids.releaseIdFrom(id))) {
      target.asInstanceOf[T]
    } else {
      readFromRepository(id)(ReadOption.withArchive()) match {
        case None =>
          throw new IllegalStateException(s"Unable to retrieve $target")
        case Some(result) =>
          if (result != target) {
            throw new IllegalStateException(s"Retrieved item $result is not equal to the $target")
          }
          result.asInstanceOf[T]
      }
    }

    if (logger.isTraceEnabled) {
      val resolveContextAsString = resolveContextCis.mkString("\n", ",\n", "\n")
      logger.trace(s"trace log for ${target}: \n ${traceLog.msg}\nresolveContext: ${resolveContextAsString}")
    }
    result
  }

  private def cacheTargetConfigurationItem[T <: ConfigurationItem](id: CiId, target: ConfigurationItem) = {
    resolveContextCis.get(id) match {
      case Some(cachedItem) if cachedItem == target => ()
      case Some(cachedItem) =>
        throw new IllegalStateException(s"CachedItem $cachedItem is not equal to desired target $target")
      case None => resolveContextCis.put(target.getId, target)
    }
  }

  def nonProxiedCachedCi(id: String): Boolean = {
    val cachedItemOption = resolveContextCis.get(id)
    cachedItemOption match {
      case None => false
      case Some(cachedItem) =>
        cachedItem match {
          case lazyConfigurationItem: LazyConfigurationItem =>
            lazyConfigurationItem.isInitialized
          case _ =>
            true
        }
    }
  }

  private def readInternal[T <: ConfigurationItem](ciId: CiId): T = {
    val id = ciId.normalized
    val res = if (nonProxiedCachedCi(id)) {
      traceLog.log(s"Loading ConfigurationItem '$id' from resolver context")
      resolveContextCis(id).asInstanceOf[T]
    } else {
      traceLog.log(s"Loading ConfigurationItem '$id' from SQL")
      try {
        val ciOption = readFromRepository(id)(ReadOption.withoutArchive())
        ciOption match {
          case Some(ci) =>
            cache(ci)
          case None =>
            logger.warn(s"Could not load ConfigurationItem '$id' from SQL")
        }
        ciOption.orNull.asInstanceOf[T]
      } catch {
        case ex: StackOverflowError =>
          throw new IllegalStateException(s"Read operation of $ciId failed. Please see captured trace for more details: ${traceLog.msg}", ex)
      }
    }
    logger.trace(traceLog.msg)
    res
  }

  private def readFromRepository(ciId: CiId)(implicit readOptions: ReadOption): Option[ConfigurationItem] = {
    ciId match {
      case AttachmentId(id) => getAttachment(id)
      case ReleaseId(id) => getRelease(id)
      case PlanItemId(id) => getPlanItem(id)
      case ConfigurationId(id) => getConfiguration(id)
      case VariableId(id) => getVariable(id)
      case ValueProviderId(id) => getValueProvider(id)
      case id =>
        logger.warn(s"Unable to load CI with Id: '$id'")
        None
    }
  }

  private def getAttachment(id: CiId): Option[Attachment] = {
    traceLog.log(s"Loading Attachment '$id' from SQL")
    val cachedCi = this.resolveContextCis.get(id)
    cachedCi.map {
      case attachment: Attachment => attachment
    } orElse {
      val attachment = releasePersistence.findAttachmentById(id)
      attachment.map(this.resolveContextCis.put(id, _))
      attachment
    }
  }

  private def getRelease(id: CiId)(implicit readOption: ReadOption): Option[Release] = {
    traceLog.log(s"Loading Release '$id'")
    val releaseId = Ids.releaseIdFrom(id)
    loadRelease(releaseId)
  }

  private def getPlanItem(id: CiId)(implicit readOption: ReadOption): Option[PlanItem] = {
    traceLog.log(s"Loading PlanItem '$id'")
    val releaseId = Ids.releaseIdFrom(id)
    loadRelease(releaseId).flatMap { release =>
      release.getAllPlanItems.asScala.find(_.getId == id)
    }
  }

  private def getVariable(id: CiId)(implicit readOption: ReadOption): Option[Variable] = {
    getFromCache(id).map {
      case variable: Variable =>
        traceLog.log(s"Loading Variable '$id' from resolver context")
        variable
    } orElse {
      val releaseId = Ids.releaseIdFrom(id)
      val release = getFromCache(releaseId) match {
        case Some(value) =>
          traceLog.log(s"Loading Variable '$id' from resolver context via release")
          Some(value.asInstanceOf[Release])
        case None =>
          traceLog.log(s"Loading Variable '$id' from SQL")
          loadRelease(releaseId)
      }
      val variable = release.flatMap(_.getVariableById(id).asScala)
      variable.map(this.resolveContextCis.put(id, _))
      variable
    }
  }

  private def getValueProvider(id: CiId)(implicit readOption: ReadOption): Option[ValueProviderConfiguration] = {
    traceLog.log(s"Loading ValueProvider '$id' from SQL")
    getVariable(Ids.getParentId(id)).flatMap(v => Option(v.getValueProvider))
  }

  private def getConfiguration(id: String): Option[BaseConfiguration] = {
    traceLog.log(s"Loading Configuration '$id' from SQL")
    configurationPersistence.findById(id).flatMap(readConfiguration)
  }

  def readConfiguration[T <: BaseConfiguration](configurationRow: ConfigurationRow): Option[T] = {
    configurationRow match {
      case (folderIdOpt, rawConfiguration) =>
        deserialize[T](rawConfiguration).map { conf =>
          folderIdOpt.fold(conf) { folderId =>
            conf.setFolderId(folderId.absolute)
            conf
          }
          cache(conf)
          conf
        }
    }
  }

  private def loadRelease(releaseId: CiId)(implicit readOption: ReadOption): Option[Release] = {
    logger.debug(s"Loading release [$releaseId]")
    if (alreadyLoadedReleases.contains(releaseId)) {
      val r: Option[ConfigurationItem] = this.resolveContextCis.get(releaseId)
      r match {
        case Some(r) => Some(r.asInstanceOf[Release])
        case None =>
          logger.warn(s"We already loaded release $releaseId but it was not found in resolved context.")
          None
      }
    } else {
      logger.debug(s"Loading release [$releaseId] from SQL")
      alreadyLoadedReleases.add(releaseId)

      val releaseRow = releasePersistence.findByReleaseId(releaseId)
      val optionalRelease = releaseRow.map {
        deserializeRelease
      }
      optionalRelease match {
        case None if readOption.useArchivedDatabase =>
          val archivedRelease = getReleaseOption(releaseId, includePreArchived = false, createCiConverter())
          archivedRelease.foreach { archivedRelease =>
            cache(archivedRelease)
            archivedRelease.setArchived(true)
          }
          archivedRelease
        case _ => optionalRelease
      }
    }
  }

  private def cache(ci: ConfigurationItem): Unit = {
    if (ci != null) {
      if (!resolveContextCis.contains(ci.getId)) {
        traceLog.log(s"Storing ${ci.getId} and it's children into resolver context")
      }
      ci match {
        case lazyConfigurationItem: LazyConfigurationItem if !lazyConfigurationItem.isInitialized =>
          resolveContextCis.getOrElseUpdate(ci.getId, {
            traceLog.log(s"Storing ${ci.getId} proxy into resolver context")
            ci
          })
        case _ =>
          cacheWithNestedCis(ci)
      }
    } else {
      val ex = new IllegalStateException("Attempted to cache null.")
      logger.warn(s"Attempted to cache null CI. Please inspect trace log: ${traceLog.msg}", ex)
    }
  }

  private def cacheWithNestedCis(ci: ConfigurationItem) = {
    CiHelper.getNestedCis(ci).asScala.foreach { cacheCi =>
      if (cacheCi.getId != null) {
        resolveContextCis.getOrElseUpdate(cacheCi.getId, {
          traceLog.log(s"Storing ${cacheCi.getId} of type ${cacheCi.getType} into resolver context")
          cacheCi
        })
      } else {
        logger.warn(s"Unable to cache nested CI of type = ${cacheCi.getType} because ID is null. Parent CI: $ci[${ci.getType}]")
      }
    }
  }

  private def getFromCache(ciId: CiId): Option[ConfigurationItem] = {
    val id = ciId.normalized
    this.resolveContextCis.get(id).orElse {
      // using endsWith here is very suspicious
      this.resolveContextCis.find(_._1.endsWith(id)).flatMap {
        case (_, ci) => Option(ci)
      }
    }
  }

  def deserializeRelease(releaseData: ReleaseRow): Release = {
    val ciConverter = createCiConverter()
    val release: Release = readFromJsonAndEnhanceWithTablesData(releaseData, ciConverter)
    cacheRelease(release)
    resolveReferences(ciConverter)
    release
  }

  def tryDeserializeRelease(releaseData: ReleaseRow): Option[Release] = {
    val ciConverter = createCiConverter()
    val optionalRelease = tryReadFromJsonAndEnhanceWithTablesData(releaseData, ciConverter)
    optionalRelease.map { release =>
      cacheRelease(release)
      resolveReferences(ciConverter)
      release
    }
  }

  private def cacheRelease(release: Release): Release = {
    cache(release)
    val folderlessReleaseId = Ids.getFolderlessId(release.getId)
    resolveContextCis.update(folderlessReleaseId, release)
    release
  }

  private def resolveReferences(ciConverter: XlrPasswordEncryptingCiConverter): Unit = {
    traceLog.increaseDepth()
    ciConverter.resolveReferences(this)
    traceLog.decreaseDepth()
  }

  def deserialize[T <: ConfigurationItem](json: String): Option[T] = {
    val ciConverter = createCiConverter()
    val ciReader = CiJson2Reader.create(json)
    val res = try {
      val ci = ciConverter.readCi(ciReader).asInstanceOf[T]
      cache(ci)
      resolveReferences(ciConverter)
      Option(ci)
    } catch {
      case e: IllegalStateException =>
        logger.error("Error reading configuration item", e)
        None
    }
    res
  }

  private def decorateRelease(releaseData: ReleaseRow, ciConverter: XlrPasswordEncryptingCiConverter)(release: Release): Release = {
    val folderId = releasePersistence.findFolderIdByReleaseId(release.getId) // release can be moved and cached value of folderId can become invalid
    val releaseCiUid = releaseData.ciUid
    release.setCiUid(releaseCiUid)
    // template logo must be loaded before we fix ids
    decorateWithTemplateMetadata(release)
    folderId.foreach(release.fixId)
    if (release.get$ciAttributes() != null) {
      release.get$ciAttributes().setScmTraceabilityDataId(Integer.valueOf(releaseData.scmDataUid))
    }
    release.setProperty("riskScore", "%03d".format(releaseData.riskScore))
    release.setProperty("totalRiskScore", "%04d".format(releaseData.totalRiskScore))

    if (release.getOriginTemplateId != null) {
      val maybeFolderId = releasePersistence.findFolderIdByReleaseId(release.getOriginTemplateId)
      maybeFolderId.foreach(folderId => release.setOriginTemplateId(Ids.formatWithFolderId(folderId, Ids.getName(release.getOriginTemplateId))))
    }

    decorateWithMutableFacets(release)

    decorateWithAttachments(release)

    // set release CiUid on all plan items
    release.getAllPlanItems.forEach(_.setReleaseUid(release.getCiUid))

    decorateTasks(release)

    if (resolveOptions.shouldResolveReferences) {
      // order matters: decorateDependencies depends on decorateTasks
      if (ReleaseLoadingFeature.batchLoadDependencies) {
        decorateDependencies(release, ciConverter)
      } else {
        oldDecorateDependencies(release, ciConverter)
      }
      prefetchConfigurations(release.getCiUid)
    }

    release
  }

  private def decorateWithTemplateMetadata(release: Release): Unit = {
    if (release.isTemplate) {
      // decorate with template metadata
      releasePersistence.findTemplateMetadata(release.getReleaseUid).foreach(templateMetadata => {
        release.setAuthor(templateMetadata.author)
        release.setLogo(templateMetadata.templateLogo)
      })
    }
  }

  private def decorateWithAttachments(release: Release): Unit = {
    val prefetchedArtifacts = releasePersistence.findAttachmentsByCiUid(release.getCiUid).map { a =>
      a.setId(release.getId + "/" + a.getId)
      a.getId -> a
    }.toMap
    release.getAttachments.asScala.foreach { attachment =>
      prefetchedArtifacts.get(attachment.getId).foreach { prefetchedArtifact =>
        attachment.setFile(prefetchedArtifact.getFile)
        attachment.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, prefetchedArtifact.getFileUri)
      }
    }
    if (null != release.getLogo) {
      prefetchedArtifacts.get(release.getLogo.getId).foreach { prefetchedArtifact =>
        release.getLogo.setFile(prefetchedArtifact.getFile)
        release.getLogo.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, prefetchedArtifact.getFileUri)
      }
    }
  }

  private def prefetchConfigurations(releaseUid: CiUid): Unit = {
    // instead of loading configurations referenced by a release one-by-one, we can load all configurations in-advance
    configurationPersistence.findAllByUid(releaseUid).foreach(readConfiguration)
  }

  private def decorateTasks(release: Release): Unit = {
    // decorate tasks with ciUid and status line
    val releaseTasks = release.getAllTasks.asScala
    val taskRowMap = taskPersistence.findByReleaseUid(release.getReleaseUid).collect {
      case t => t.taskId -> t
    }.toMap
    for {
      task <- releaseTasks
      row <- taskRowMap.get(getFolderlessId(task.getId))
    } {
      task.setCiUid(row.ciUid)
      if (task.hasProperty("statusLine")) {
        task.setProperty("statusLine", row.statusLine)
      }
    }
  }

  private def decorateWithMutableFacets(release: Release): Unit = {
    val facets = facetPersistence.findByParentId(release.getId).filter(_.content.isRegistered).flatMap { row =>
      deserialize[Facet](row.content.withoutUnknownProps)
        .map { facet =>
          facet.setTargetId(row.targetId)
          facet
        }
    }.filter(!_.getType.isSubTypeOf(Type.valueOf(classOf[TaskReportingRecord])))

    release.getAllTasks.asScala.foreach { task =>
      task.setFacets(new util.ArrayList(facets.filter(_.getTargetId == task.getId).asJava))
    }
  }

  lazy val targetProperty = Type.valueOf(classOf[Dependency]).getDescriptor.getPropertyDescriptor("target")

  private def decorateDependencies(release: Release, ciConverter: XlrPasswordEncryptingCiConverter): Unit = {
    val dependencies = release.getAllGates.asScala.flatMap { gt =>
      gt.dependencies.map(_.setGateTask(gt)) // link deps to gate task so it would not cause NPE below
      gt.dependencies
    }
    logger.trace(s"Checking ${dependencies.size} dependencies loaded from release JSON ${release.getId} against the database")
    val dependenciesWithTargetId = dependencies.filter(d => !d.isArchived && d.getTargetId != null && Ids.isPlanItemId(d.getTargetId))
    // 1. load dependencies and their targets from that database with 1 query
    val depRows = dependencyPersistence.findByGateReleaseUid(release.getReleaseUid)
    // r.targetRow.getTargetId is fishy - to many options
    // dependency id can be the same on different tasks, so key should be dependency_id name AND GATE_TASK_UID
    val targetIdsByDependencyKey = depRows.map(r => (DependencyKey(r), r.targetRow.getTargetId.get)).toMap
    dependenciesWithTargetId.foreach { dependency =>
      val dependencyKey = DependencyKey(dependency)
      val maybeTargetId = targetIdsByDependencyKey.get(dependencyKey)
      maybeTargetId.map { targetId =>
        // 2. if target found in database does not match target defined on the dependency replace it with the target found in the database dependency
        if (dependency.getTargetId != targetId) {
          logger.trace(s"Overriding dependency ${dependency.getId} target ID ${dependency.getTargetId} with value from the database: $targetId")
          dependency.setTargetId(targetId)
          // Replace the ciConverter reference which will be used for resolution later
          ciConverter.getReferences.removeIf(ref => ref.getCi.eq(dependency) && ref.getProperty == targetProperty)
          if (Ids.hasReleaseId(targetId)) {
            // try to resolve only resolvable release plan items (ie skip Phase123/Task123)
            ciConverter.getReferences.add(new CiReference(dependency, targetProperty, targetId))
          }
        }
        val targetIsNotVariable = !VariableHelper.containsVariables(targetId)
        val targetIsInsideRelease = Ids.hasReleaseId(targetId)
        val targetIsNotAlreadyReferenced = !ciConverter.getReferences.asScala.exists(ref => ref.getCi.eq(dependency) && ref.getProperty == targetProperty)
        if (targetIsNotVariable && targetIsInsideRelease && targetIsNotAlreadyReferenced) {
          ciConverter.getReferences.add(new CiReference(dependency, targetProperty, targetId))
        }
      }
    }
  }

  private def oldDecorateDependencies(release: Release, ciConverter: XlrPasswordEncryptingCiConverter): Unit = {
    val dependencies = release.getAllGates.asScala.flatMap(_.dependencies)
    logger.trace(s"Checking ${dependencies.size} dependencies loaded from release JSON ${release.getId} against the database")
    dependencies
      .filter(d => !d.isArchived && d.getTargetId != null && Ids.isPlanItemId(d.getTargetId))
      .foreach(decorateDependency(ciConverter))
  }

  // scalastyle:off cyclomatic.complexity
  private def decorateDependency(ciConverter: XlrPasswordEncryptingCiConverter)(dependency: Dependency): Unit = {
    val taskCiUid: Option[CiUid] = Try {
      Option(dependency.getGateTask)
        .map(_.getCiUid)
        .orElse(taskPersistence.getTaskUidById(Ids.getParentId(dependency.getId)))
    }.recover {
      case e => logger.error(s"Exception when getting task CiUid for dependency ${dependency.getId}: ", e)
        None
    }.get

    val targetId: Option[String] = Try {
      taskCiUid
        .flatMap(gateTaskCiUid => dependencyPersistence.getTargetId(dependency.getId, gateTaskCiUid))
    }.recover {
      case e =>
        logger.warn(s"Got exception when fetching dependency targetId dependency ${dependency.getId}, gateTaskCiUid = ${taskCiUid}", e)
        None
    }.get
    targetId
      .orElse {
        logger.warn(s"Could not find dependency ${dependency.getId} in the database, targetId = [${dependency.getTargetId}]")
        None
      }
      .map { targetId =>
        if (targetId != dependency.getTargetId) {
          logger.trace(s"Overriding dependency ${dependency.getId} target ID ${dependency.getTargetId} with value from the database: $targetId")
          if (!Ids.isInRelease(targetId) && Ids.isReleaseId(targetId)) {
            logger.error(s"Dependency ${dependency.getId} target ${dependency.getTargetId} will be replaced by $targetId" +
              s" that does not point to a valid plan item ID. Please report a bug.")
          }
          dependency.setTargetId(targetId)

          // Replace the ciConverter reference which will be used for resolution later
          ciConverter.getReferences.removeIf(ref => ref.getCi.eq(dependency) && ref.getProperty == targetProperty)
          if (Ids.hasReleaseId(targetId)) {
            // try to resolve only resolvable release plan items (ie skip Phase123/Task123)
            ciConverter.getReferences.add(new CiReference(dependency, targetProperty, targetId))
          }
        }
        targetId
      }
      .foreach(targetId => {
        if (!VariableHelper.containsVariables(targetId) &&
          Ids.hasReleaseId(targetId) &&
          !ciConverter.getReferences.asScala.exists(ref => ref.getCi.eq(dependency) && ref.getProperty == targetProperty)
        ) {
          ciConverter.getReferences.add(new CiReference(dependency, targetProperty, targetId))
        }
      })
  }

  private def readFromJsonAndEnhanceWithTablesData(releaseData: ReleaseRow, ciConverter: XlrPasswordEncryptingCiConverter): Release = {
    val ciReader = CiJson2Reader.create(releaseData.json.withoutUnknownTypes)
    val release = ciConverter.readCi(ciReader).asInstanceOf[Release]
    decorateRelease(releaseData, ciConverter)(release)
  }

  private def tryReadFromJsonAndEnhanceWithTablesData(releaseData: ReleaseRow, ciConverter: XlrPasswordEncryptingCiConverter): Option[Release] = {
    val ciReader = CiJson2Reader.create(releaseData.json.withoutUnknownTypes)
    Try(ciConverter.readCi(ciReader).asInstanceOf[Release])
      .map(decorateRelease(releaseData, ciConverter))
      .map(Option(_))
      .recoverWith {
        case e: IllegalArgumentException =>
          logger.error(s"Unrecoverable error detected. Cannot parse release ${
            releaseData.releaseId
          } from json", e)
          Success(None)
        case e: SerializationException =>
          logger.error(s"Error deserializing release ${
            releaseData.releaseId
          } from json. Probably some property value doesn't conform it's type", e.getCause)
          Success(None)
      }.get
  }

  class MessageLogger {
    private val logBuffer: mutable.Buffer[String] = mutable.ArrayBuffer.empty
    private var depth: Int = 0
    val prefix = "    "

    def increaseDepth(): Unit = depth = depth + 1

    def decreaseDepth(): Unit = depth = depth - 1


    def log(msg: String): Unit = {
      logBuffer += s"${prefix * depth}$msg"
    }

    def msg: String = {
      logBuffer.mkString("\n", "\n", "\n")
    }
  }

  private case class ReadOption(useArchivedDatabase: Boolean)

  private object ReadOption {
    def withoutArchive(): ReadOption = ReadOption(false)

    def withArchive(): ReadOption = ReadOption(true)
  }

  case class DependencyKey(dependencyId: String, gateTaskId: String)

  object DependencyKey {
    def apply(dr: DependencyRow): DependencyKey = {
      DependencyKey(dr.dependencyId, dr.taskId)
    }

    def apply(dependency: Dependency): DependencyKey = {
      val dependencyId = Ids.getName(dependency.getId)
      val normalizedTaskId = dependency.getGateTask.getId.shortId
      DependencyKey(dependencyId, normalizedTaskId)
    }
  }
}
