package com.xebialabs.xlrelease.repository.sql.query

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.api.v1.filter.WorkflowFilters
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.domain.{ReleaseKind, TemplateLogo}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{SEPARATOR, getName}
import com.xebialabs.xlrelease.repository.query._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema.{ROLES, ROLE_PERMISSIONS, ROLE_PRINCIPALS, ROLE_ROLES}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.utils.FolderId
import com.xebialabs.xlrelease.view.WorkflowOverview
import org.joda.time.DateTime
import org.springframework.jdbc.core.RowMapper
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate
import org.springframework.util.{StreamUtils, StringUtils}

import java.nio.charset.StandardCharsets
import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.Using

trait WorkflowQueryBuilder {
  def from(workflowFilters: WorkflowFilters): FiltersQueryBuilder[WorkflowFilters, WorkflowOverview]
}

object WorkflowQueryBuilder {
  def apply(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate): WorkflowQueryBuilder = new SqlWorkflowQueryBuilder(dialect, namedTemplate)
}

class SqlWorkflowQueryBuilder(dialect: Dialect, namedTemplate: NamedParameterJdbcTemplate) extends WorkflowQueryBuilder {
  override def from(workflowFilters: WorkflowFilters): FiltersQueryBuilder[WorkflowFilters, WorkflowOverview] = {
    new SqlWorkflowFiltersQueryBuilder(dialect, namedTemplate).from(workflowFilters)
  }
}

class SqlWorkflowFiltersQueryBuilder(val dialect: Dialect,
                                     val namedTemplate: NamedParameterJdbcTemplate)
  extends FiltersQueryBuilder[WorkflowFilters, WorkflowOverview]
    with FilterQueryBuilderSupport[WorkflowFilters, WorkflowOverview] {

  private def rowMapper: RowMapper[WorkflowOverview] = (rs: ResultSet, _: Int) => {
    val rawReleaseId = rs.getString(RELEASES.RELEASE_ID)
    val rawFolderId = rs.getString(FOLDERS.FOLDER_ID)
    val rawFolderPath = rs.getString(FOLDERS.FOLDER_PATH)
    val folderId = s"$rawFolderPath$SEPARATOR$rawFolderId".dropWhile(_ == '/')
    val releaseId = s"$rawFolderPath$SEPARATOR$rawFolderId$SEPARATOR$rawReleaseId".dropWhile(_ == '/')

    val workflowOverview = new WorkflowOverview()
    workflowOverview.setCiUid(rs.getInt(RELEASES.CI_UID))
    workflowOverview.setId(FolderId(releaseId).absolute)
    workflowOverview.setTitle(rs.getString(RELEASES.RELEASE_TITLE))
    workflowOverview.setAuthor(rs.getString(TEMPLATE_METADATA.AUTHOR))

    val descriptionStream = rs.getBinaryStream(TEMPLATE_METADATA.DESCRIPTION)
    val description = if (descriptionStream == null) {
      null
    } else {
      Using.resource(descriptionStream) { is =>
        val bytes = StreamUtils.copyToByteArray(is)
        new String(bytes, StandardCharsets.UTF_8)
      }
    }
    workflowOverview.setDescription(description)

    val logoStream = rs.getBinaryStream(TEMPLATE_METADATA.LOGO_DATA)
    val templateLogo = if (logoStream == null) {
      null
    } else {
      Using.resource(logoStream) { is =>
        val logoBytes = StreamUtils.copyToByteArray(is)
        val logoJson = new String(logoBytes, StandardCharsets.UTF_8)
        val logo = CiSerializerHelper.deserialize(logoJson, null).asInstanceOf[TemplateLogo]
        logo.setId(FolderId(s"$releaseId/${getName(logo.getId)}").absolute)
        logo
      }
    }
    workflowOverview.setLogo(templateLogo)

    workflowOverview.setFolderId(FolderId(if (folderId == null) Ids.ROOT_FOLDER_ID else folderId).absolute)
    workflowOverview.setFolderTitle(rs.getString(FOLDERS.NAME))

    val kind = rs.getString(SCM_DATA.SCM_KIND)
    val commit = rs.getString(SCM_DATA.SCM_COMMIT)
    val author = rs.getString(SCM_DATA.SCM_AUTHOR)
    val date = rs.getTimestamp(SCM_DATA.SCM_DATE)
    val message = rs.getString(SCM_DATA.SCM_MESSAGE)
    val remote = rs.getString(SCM_DATA.SCM_REMOTE)
    val fileName = rs.getString(SCM_DATA.SCM_FILENAME)
    val scmData = new SCMTraceabilityData(kind, commit, author, new DateTime(date), message, remote, fileName)

    workflowOverview.setScmTraceabilityData(scmData)

    workflowOverview
  }

  private lazy val queryTemplate =
    s"""
       |SELECT
       |  rel.${RELEASES.CI_UID},
       |  rel.${RELEASES.RELEASE_ID},
       |  rel.${RELEASES.RELEASE_TITLE},
       |  xtm.${TEMPLATE_METADATA.AUTHOR},
       |  xtm.${TEMPLATE_METADATA.DESCRIPTION},
       |  xtm.${TEMPLATE_METADATA.LOGO_DATA},
       |  xsd.${SCM_DATA.SCM_KIND},
       |  xsd.${SCM_DATA.SCM_COMMIT},
       |  xsd.${SCM_DATA.SCM_AUTHOR},
       |  xsd.${SCM_DATA.SCM_DATE},
       |  xsd.${SCM_DATA.SCM_MESSAGE},
       |  xsd.${SCM_DATA.SCM_REMOTE},
       |  xsd.${SCM_DATA.SCM_FILENAME},
       |  f.${FOLDERS.FOLDER_ID},
       |  f.${FOLDERS.FOLDER_PATH},
       |  f.${FOLDERS.NAME}
       |FROM
       |  ${RELEASES.TABLE} rel
       |  LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = rel.${RELEASES.FOLDER_CI_UID}
       |  LEFT JOIN ${TEMPLATE_METADATA.TABLE} xtm on rel.${RELEASES.CI_UID} = xtm.${TEMPLATE_METADATA.RELEASE_UID}
       |  LEFT JOIN ${SCM_DATA.TABLE} xsd on rel.${RELEASES.SCM_DATA} = xsd.${SCM_DATA.SCM_DATA_ID}
       |$joinClause
       |$whereClause
       |$orderClause
    """.stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(NL)

  private lazy val totalQueryTemplate =
    s"""
       |SELECT COUNT(1)
       |FROM
       |  ${RELEASES.TABLE} rel
       |  LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = rel.${RELEASES.FOLDER_CI_UID}
       |  LEFT JOIN ${TEMPLATE_METADATA.TABLE} xtm on rel.${RELEASES.CI_UID} = xtm.${TEMPLATE_METADATA.RELEASE_UID}
       |  LEFT JOIN ${SCM_DATA.TABLE} xsd on rel.${RELEASES.SCM_DATA} = xsd.${SCM_DATA.SCM_DATA_ID}
       |$whereClause
    """.stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")

  private lazy val GET_RELEASE_UID_BY_CATEGORIES =
    s"""
       |SELECT xrf.${RELEASE_CATEGORY_REFS.sourceColumn}
       |  FROM
       |    ${RELEASE_CATEGORY_REFS.TABLE} xrf
       |    JOIN ${CATEGORIES.TABLE} xc on xrf.${RELEASE_CATEGORY_REFS.targetColumn} = xc.${CATEGORIES.CI_UID}
       |  WHERE
       |    xc.${CATEGORIES.CATEGORY} IN (:categories)
       |""".stripMargin

  private def selectCiIdsWithPermission(roleIds: Iterable[String], permissionsParameterName: String): String = {
    s"""|SELECT
        | DISTINCT teamPermissions.${ROLE_PERMISSIONS.ciId}
        |FROM
        | ${ROLES.TABLE} teams
        | LEFT JOIN ${ROLE_ROLES.TABLE} teamRoles ON teams.${ROLES.id} = teamRoles.${ROLE_ROLES.roleId}
        | LEFT JOIN ${ROLE_PRINCIPALS.TABLE} teamPrincipals ON teams.${ROLES.id} = teamPrincipals.${ROLE_PRINCIPALS.roleId}
        | JOIN ${ROLE_PERMISSIONS.TABLE} teamPermissions ON teams.${ROLES.id} = teamPermissions.${ROLE_PERMISSIONS.roleId}
        |WHERE
        | teamPermissions.${ROLE_PERMISSIONS.permissionName} IN (:$permissionsParameterName)
        | AND (
        |   teamPrincipals.${ROLE_PRINCIPALS.principalName} IN (:principalNames)
        |   ${if (roleIds.nonEmpty) s"OR teamRoles.${ROLE_ROLES.memberRoleId} IN (:roleIds)" else ""}
        | )""".stripMargin.linesIterator.filter(_.trim.nonEmpty).mkString(s"$NL")
  }

  override def from(filter: WorkflowFilters): SelfType = {
    equal(s"rel.${RELEASES.KIND}", RELEASES.KIND, ReleaseKind.WORKFLOW.value())
    if (filter.getStatuses.asScala.nonEmpty) {
      whereClauses += s"rel.${RELEASES.STATUS} IN (:statuses)"
      queryParams += "statuses" -> filter.getStatuses.asScala.map(_.value()).toSet.asJava
    }

    if (!filter.getCategories.isEmpty) {
      whereClauses += s"rel.${RELEASES.CI_UID} IN ($GET_RELEASE_UID_BY_CATEGORIES)"
      queryParams += "categories" -> filter.getCategories
    }
    if (StringUtils.hasText(filter.getFolderId)) {
      equal(s"f.${FOLDERS.FOLDER_ID}", FOLDERS.FOLDER_ID, getName(filter.getFolderId))
    }
    if (StringUtils.hasText(filter.getAuthor)) {
      like(s"xtm.${TEMPLATE_METADATA.AUTHOR}", TEMPLATE_METADATA.AUTHOR, filter.getAuthor)
    }
    if (StringUtils.hasText(filter.getSearchInput)) {
      like(s"rel.${RELEASES.RELEASE_TITLE}", RELEASES.RELEASE_TITLE, filter.getSearchInput)
    }
    this
  }

  override def withPermissions(permissions: Seq[Permission], principals: Iterable[String], roleIds: Iterable[String]): SelfType = {
    whereClauses +=
      s"""rel.${RELEASES.SECURITY_UID} IN (
         |  ${selectCiIdsWithPermission(roleIds, "permissionNames")}
         |)
      """.stripMargin
    queryParams += "permissionNames" -> permissions.map(permission => permission.getPermissionName).toList.asJava
    queryParams += "principalNames" -> principals.toList.asJava
    queryParams += "roleIds" -> roleIds.toList.asJava
    this
  }

  override def build(): PageableQuery[WorkflowOverview] = {
    remapSortOrderParams()
    buildSortOrderClause(RELEASES.RELEASE_TITLE)
    val resultsQueryString = pageableQuery(queryTemplate)
    val resultsQuery = new SqlListQuery[WorkflowOverview](namedTemplate, resultsQueryString, queryParams.toMap, rowMapper)
    val countQueryString = totalQueryTemplate
    val totalCountQuery = new SqlQuery[Long](namedTemplate, countQueryString, queryParams.toMap, (rs, _) => rs.getLong(1))
    new SqlPageableQuery[WorkflowOverview](namedTemplate, this.pageable, resultsQuery, totalCountQuery)
  }

  private def remapSortOrderParams(): Unit = {
    this.withSortParameters(
      RELEASES.RELEASE_TITLE -> s"rel.${RELEASES.RELEASE_TITLE}"
    )
  }
}
