package com.xebialabs.xlrelease.security

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.SecuritySchema._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.security.SqlUserGroupRepository._
import com.xebialabs.xlrelease.security.sql.Diff
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.util.Assert
import org.springframework.util.StringUtils.hasText

import scala.jdk.CollectionConverters._

@IsTransactional
class SqlUserGroupRepository(ciIdService: CiIdService,
                             val jdbcTemplate: JdbcTemplate,
                             val dialect: Dialect)
  extends UserGroupRepository with PersistenceSupport with Logging {

  @Timed
  @IsReadOnly
  override def findGroupsForUser(username: String): Set[String] = {
    Assert.hasText(username, "username cannot be empty")
    logger.trace(s"Finding groups for user '$username'")
    findMany(
      sqlQuery(STMT_FIND_GROUPS_FOR_USER,
        params(GROUP_PRINCIPALS.PRINCIPAL_NAME -> username.toLowerCase),
        rs => rs.getString(GROUPS.NAME)
      )
    ).toSet
  }

  @Timed
  override def createGroup(groupName: String): String = {
    Assert.hasText(groupName, "groupName cannot be empty")
    logger.trace(s"Creating new group '$groupName'")
    val groupId: String = generateGroupId()
    sqlInsert(
      STMT_INSERT_GROUP,
      params(
        GROUPS.ID -> groupId,
        GROUPS.NAME -> groupName
      )
    )
    groupId
  }

  @Timed
  override def createGroups(groupNames: Set[String]): Set[String] = {
    Assert.notEmpty(groupNames.asJava, "groupNames cannot be empty")
    val nonEmptyGroupNames = groupNames.filter(hasText)
    nonEmptyGroupNames.grouped(100).flatMap { chunkedGroupNames =>
      val groups = chunkedGroupNames.map(Group(generateGroupId(), _))
      sqlBatch(
        STMT_INSERT_GROUP,
        groups.map { group =>
          params(
            GROUPS.ID -> group.id,
            GROUPS.NAME -> group.name
          )
        }
      )
      groups.map(_.id)
    }.toSet
  }

  @Timed
  override def deleteGroup(groupName: String): Unit = {
    Assert.hasText(groupName, "groupName cannot be empty")
    logger.trace(s"Deleting group '$groupName'")
    findGroupId(groupName) match {
      case Some(groupId) =>
        sqlExec(STMT_DELETE_USER_GROUP_MEMBERSHIP_BY_GROUP, params(GROUP_PRINCIPALS.GROUP_ID -> groupId), _.execute())
        sqlExec(STMT_DELETE_GROUP, params(GROUPS.NAME -> groupName), _.execute())
      case None =>
        logger.trace(s"Group '$groupName' not found")
    }
  }

  @Timed
  override def addUserToGroup(username: String, groupName: String): Unit = {
    Assert.hasText(username, "username cannot be empty")
    Assert.hasText(groupName, "groupName cannot be empty")
    logger.trace(s"Adding user '$username' to group '$groupName'")
    val groupId: String = findGroupId(groupName) match {
      case Some(id) => id
      case None => createGroup(groupName)
    }

    sqlInsert(
      STMT_INSERT_USER_GROUP_MEMBERSHIP,
      params(
        GROUP_PRINCIPALS.GROUP_ID -> groupId,
        GROUP_PRINCIPALS.PRINCIPAL_NAME -> username.toLowerCase
      )
    )
  }

  @Timed
  override def addUserToGroups(username: String, groupNames: Set[String]): Unit = {
    Assert.hasText(username, "username cannot be empty")
    Assert.notEmpty(groupNames.asJava, "groupNames cannot be empty")
    logger.trace(s"Adding user '$username' to groups '$groupNames'")
    val existingGroups = findGroups(groupNames)
    val newGroups = groupNames.filterNot(existingGroups.map(_.name))
    val groupIds: Set[String] = if (newGroups.nonEmpty) {
      val newGroupIds = createGroups(newGroups)
      newGroupIds ++ existingGroups.map(_.id)
    } else {
      existingGroups.map(_.id)
    }

    groupIds.grouped(100).foreach { chunkedGroupIds =>
      sqlBatch(
        STMT_INSERT_USER_GROUP_MEMBERSHIP,
        chunkedGroupIds.map { groupId =>
          params(
            GROUP_PRINCIPALS.GROUP_ID -> groupId,
            GROUP_PRINCIPALS.PRINCIPAL_NAME -> username.toLowerCase
          )
        }
      )
    }
  }

  @Timed
  override def removeUserFromGroup(username: String, groupName: String): Unit = {
    Assert.hasText(username, "username cannot be empty")
    Assert.hasText(groupName, "groupName cannot be empty")
    logger.trace(s"Removing user '$username' from group '$groupName'")
    findGroupId(groupName) match {
      case Some(groupId) =>
        sqlExec(
          STMT_DELETE_USER_GROUP_MEMBERSHIP,
          params(
            GROUP_PRINCIPALS.GROUP_ID -> groupId,
            GROUP_PRINCIPALS.PRINCIPAL_NAME -> username.toLowerCase
          ),
          _.execute())
      case None =>
        logger.trace(s"Group '$groupName' not found")
    }
  }

  @Timed
  override def removeUserFromGroups(username: String, groupNames: Set[String]): Unit = {
    Assert.hasText(username, "username cannot be empty")
    Assert.notEmpty(groupNames.asJava, "groupNames cannot be empty")
    logger.trace(s"Removing user '$username' from groups '$groupNames'")
    val groupIds = findGroups(groupNames).map(_.id)
    groupIds.grouped(100).foreach { chunkedGroupIds =>
      sqlExec(
        STMT_DELETE_USER_GROUPS_MEMBERSHIP,
        params(
          GROUP_PRINCIPALS.PRINCIPAL_NAME -> username.toLowerCase,
          GROUP_PRINCIPALS.GROUP_ID -> chunkedGroupIds.asJava
        ),
        _.execute()
      )
    }
  }

  @Timed
  override def updateGroupsMembershipForUser(username: String, groupNames: Set[String]): Unit = {
    val existingGroupsMembership = findGroupsForUser(username)
    val diff = Diff(existingGroupsMembership, groupNames)

    if (diff.newEntries.nonEmpty) {
      if (diff.newEntries.size == 1) {
        addUserToGroup(username, diff.newEntries.head)
      } else {
        addUserToGroups(username, diff.newEntries)
      }
    }

    if (diff.deletedEntries.nonEmpty) {
      if (diff.deletedEntries.size == 1) {
        removeUserFromGroup(username, diff.deletedEntries.head)
      } else {
        removeUserFromGroups(username, diff.deletedEntries)
      }
    }
  }

  private def findGroupId(groupName: String): Option[String] = {
    findOne(
      sqlQuery(STMT_FIND_GROUP_ID,
        params(GROUPS.NAME -> groupName),
        rs => rs.getString(GROUPS.ID)
      )
    )
  }

  private def findGroups(groupNames: Set[String]): Set[Group] = {
    groupNames.filter(hasText).grouped(100).flatMap { chunkedGroupNames =>
      sqlQuery(STMT_FIND_GROUPS,
        params(GROUPS.NAME -> chunkedGroupNames.asJava),
        rs => Group(rs.getString(GROUPS.ID), rs.getString(GROUPS.NAME))
      )
    }.toSet
  }

  private def generateGroupId(): String = {
    Ids.getName(ciIdService.getUniqueId("Group", ""))
  }
}

object SqlUserGroupRepository {
  private val STMT_FIND_GROUPS_FOR_USER: String =
    s"""SELECT g.${GROUPS.NAME}
       | FROM ${GROUPS.TABLE} g
       | JOIN ${GROUP_PRINCIPALS.TABLE} gp ON g.${GROUPS.ID} = gp.${GROUP_PRINCIPALS.GROUP_ID}
       | WHERE gp.${GROUP_PRINCIPALS.PRINCIPAL_NAME} = :${GROUP_PRINCIPALS.PRINCIPAL_NAME}
       |""".stripMargin

  private val STMT_INSERT_GROUP: String =
    s"""INSERT INTO ${GROUPS.TABLE}
       | (${GROUPS.ID}, ${GROUPS.NAME})
       | VALUES (:${GROUPS.ID}, :${GROUPS.NAME})
       |""".stripMargin

  private val STMT_DELETE_GROUP: String =
    s"DELETE FROM ${GROUPS.TABLE} WHERE ${GROUPS.NAME} = :${GROUPS.NAME}"

  private val STMT_DELETE_USER_GROUP_MEMBERSHIP_BY_GROUP: String =
    s"""DELETE FROM ${GROUP_PRINCIPALS.TABLE}
       | WHERE ${GROUP_PRINCIPALS.GROUP_ID} = :${GROUP_PRINCIPALS.GROUP_ID}
       |""".stripMargin

  private val STMT_INSERT_USER_GROUP_MEMBERSHIP: String =
    s"""INSERT INTO ${GROUP_PRINCIPALS.TABLE}
       | (${GROUP_PRINCIPALS.GROUP_ID}, ${GROUP_PRINCIPALS.PRINCIPAL_NAME})
       | VALUES (:${GROUP_PRINCIPALS.GROUP_ID}, :${GROUP_PRINCIPALS.PRINCIPAL_NAME})
       |""".stripMargin

  private val STMT_DELETE_USER_GROUP_MEMBERSHIP: String =
    s"""DELETE FROM ${GROUP_PRINCIPALS.TABLE}
       | WHERE ${GROUP_PRINCIPALS.GROUP_ID} = :${GROUP_PRINCIPALS.GROUP_ID}
       | AND ${GROUP_PRINCIPALS.PRINCIPAL_NAME} = :${GROUP_PRINCIPALS.PRINCIPAL_NAME}
       |""".stripMargin

  private val STMT_DELETE_USER_GROUPS_MEMBERSHIP: String =
    s"""DELETE FROM ${GROUP_PRINCIPALS.TABLE}
       | WHERE ${GROUP_PRINCIPALS.PRINCIPAL_NAME} = :${GROUP_PRINCIPALS.PRINCIPAL_NAME}
       | AND ${GROUP_PRINCIPALS.GROUP_ID} IN (:${GROUP_PRINCIPALS.GROUP_ID})
       |""".stripMargin

  private val STMT_FIND_GROUP_ID: String =
    s"SELECT ${GROUPS.ID} FROM ${GROUPS.TABLE} WHERE ${GROUPS.NAME} = :${GROUPS.NAME}"

  private val STMT_FIND_GROUPS: String =
    s"SELECT ${GROUPS.ID}, ${GROUPS.NAME} FROM ${GROUPS.TABLE} WHERE ${GROUPS.NAME} IN (:${GROUPS.NAME})"

  private case class Group(id: String, name: String)
}
