package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, Type}
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.{Configuration, CustomScriptTask, Task}
import com.xebialabs.xlrelease.repository.Ids
import org.springframework.stereotype.Component

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
class ServerConnectionMigrator(sharedConfigurationService: SharedConfigurationService) {
  def migrateProperties(undefinedSource: Task, undefinedTarget: Task): Unit = {
    val source = getTrueTask(undefinedSource)
    val target = getTrueTask(undefinedTarget)

    getServerPropertyNamesWithAliases(source).foreach { serverPropertyNameWithAliases =>
      val maybeSourceServerTitle = extractServerTitle(source, serverPropertyNameWithAliases)
      val maybeTargetServerDescriptor = extractServerDescriptor(target, serverPropertyNameWithAliases)
      for {
        sourceServerTitle <- maybeSourceServerTitle
        targetServerDescriptor <- maybeTargetServerDescriptor
      } {
        findAndAssignServerDescriptor(target, sourceServerTitle, targetServerDescriptor)
      }
    }
  }

  private def getTrueTask(source: Task): BaseConfigurationItem = {
    source match {
      case task: CustomScriptTask => task.getPythonScript
      case _ => source
    }
  }

  private def getServerPropertyNamesWithAliases(sourceTask: BaseConfigurationItem): Set[PropertyNameAliasesPair] = {
    val serverPropertyDescriptors = findAllServerPropertyDescriptors(sourceTask)
    val serverPropertyNamesWithAliases = serverPropertyDescriptors.map(pd => PropertyNameAliasesPair(pd.getName, pd.getAliases.asScala.toSet)).toSet
    serverPropertyNamesWithAliases
  }

  private def extractServerTitle[T <: BaseConfigurationItem](sourceTask: T, serverPropertyNameAliasesPair: PropertyNameAliasesPair): Option[String] = {
    Try(sourceTask.getProperty(serverPropertyNameAliasesPair.name).asInstanceOf[Configuration]).toOption
      .flatMap(server => Option(server).map(_.getTitle))
  }

  private def extractServerDescriptor[T <: BaseConfigurationItem](targetTask: T, serverPropertyNameAliasesPair: PropertyNameAliasesPair): Option[PropertyDescriptor] = {
    val maybeServerPropertyName = determineServerPropertyName(targetTask, serverPropertyNameAliasesPair)
    val maybeServerDescriptor = maybeServerPropertyName.flatMap { propertyName =>
      Option(targetTask.getType)
        .flatMap { targetType =>
          Option(targetType.getDescriptor.getPropertyDescriptor(propertyName))
        }
    }
    maybeServerDescriptor
  }

  private def determineServerPropertyName[T <: BaseConfigurationItem](targetTask: T, serverPropertyNameAliasesPair: PropertyNameAliasesPair): Option[String] = {
    val pds = findAllServerPropertyDescriptors(targetTask)
    val serverPropertyNames = pds.map(_.getName).toSet
    val serverPropertyNamesWithNameMatchingAliases = pds.filter(_.getAliases.contains(serverPropertyNameAliasesPair.name)).map(_.getName)
    val serverPropertyNamesWithAliasesMatchingName = pds.filter(pd => serverPropertyNameAliasesPair.aliases.contains(pd.getName)).map(_.getName)
    val serverPropertyNamesWithAliasesMatchingAliases = pds.filter(_.getAliases.asScala.exists(serverPropertyNameAliasesPair.aliases)).map(_.getName)

    val maybeServerPropertyName =
      if (serverPropertyNames.contains(serverPropertyNameAliasesPair.name)) Some(serverPropertyNameAliasesPair.name)
      else if (serverPropertyNamesWithNameMatchingAliases.size == 1) Some(serverPropertyNamesWithNameMatchingAliases.head)
      else if (serverPropertyNamesWithAliasesMatchingName.size == 1) Some(serverPropertyNamesWithAliasesMatchingName.head)
      else if (serverPropertyNamesWithAliasesMatchingAliases.size == 1) Some(serverPropertyNamesWithAliasesMatchingAliases.head)
      else None

    maybeServerPropertyName
  }

  private def findAllServerPropertyDescriptors[T <: BaseConfigurationItem](targetTask: T): Iterable[PropertyDescriptor] = {
    val pds = targetTask.getType.getDescriptor.getPropertyDescriptors.asScala
    val serverPropertyDescriptors = pds.filter { pd =>
      val itemType = pd.getReferencedType
      itemType != null && itemType.isSubTypeOf(Type.valueOf(classOf[Configuration]))
    }
    serverPropertyDescriptors
  }

  private def findAndAssignServerDescriptor(target: BaseConfigurationItem, sourceServerTitle: String, targetServerDescriptor: PropertyDescriptor): Unit = {
    val targetType = targetServerDescriptor.getReferencedType
    val folderId = Ids.findFolderId(target.getId)
    val targetServerConfigurations = sharedConfigurationService.searchByTypeAndTitle(targetType, null, folderId, false).asScala.toSeq
    val groupedTargetServerConfigurations: Map[String, Seq[Configuration]] = targetServerConfigurations.groupBy {
      case configuration if configuration.getFolderId != null => configuration.getFolderId
      case _ => Folder.ROOT_FOLDER_ID
    }

    val folderPathHierarchy = collectFolderPathHierarchy(folderId)

    // First, cycle through the folder hierarchy to find the closest target connection with the same name
    folderPathHierarchy.foreach { folderPath =>
      val targetServerConfigurations = groupedTargetServerConfigurations.getOrElse(folderPath, Seq())
      val maybeTargetServerConfig = findTargetServerConfigurationWithSameTitle(targetServerConfigurations, sourceServerTitle)
      maybeTargetServerConfig.foreach { targetServerConfig =>
        targetServerDescriptor.set(target, targetServerConfig)
        return // Stop looking if a target connection with the same name is found
      }
    }

    // Second, cycle through the folder hierarchy to find the closest target connection when one with the same name is not found
    folderPathHierarchy.foreach { folderPath =>
      val targetServerConfigurations = groupedTargetServerConfigurations.getOrElse(folderPath, Seq())
      val multipleConnectionsFoundAtSameLevel = targetServerConfigurations.size > 1
      if (multipleConnectionsFoundAtSameLevel) return
      val maybeTargetServerConfig = findTargetServerConfiguration(targetServerConfigurations)
      maybeTargetServerConfig.foreach { targetServerConfig =>
        targetServerDescriptor.set(target, targetServerConfig)
        return // Stop looking if a target connection is found
      }
    }
  }

  private def collectFolderPathHierarchy(folderId: String): ListBuffer[String] = {
    val folderPathHierarchy = ListBuffer[String]()
    var currentFolder = folderId
    while (!Ids.isRoot(currentFolder)) {
      folderPathHierarchy += currentFolder
      currentFolder = Ids.getParentId(Ids.findFolderId(currentFolder))
    }
    folderPathHierarchy += currentFolder
    folderPathHierarchy
  }

  private def findTargetServerConfigurationWithSameTitle(targetServerConfigurations: Seq[Configuration], sourceServerTitle: String): Option[Configuration] = {
    val filtered = targetServerConfigurations.filter(config => config.getTitle == sourceServerTitle)
    if (filtered.length == 1) {
      filtered.headOption
    } else {
      None
    }
  }

  private def findTargetServerConfiguration(targetServerConfigurations: Seq[Configuration]): Option[Configuration] = {
    if (targetServerConfigurations.length == 1)
      targetServerConfigurations.headOption
    else
      None
  }
}
