package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence

import scala.jdk.CollectionConverters._

private[sql] trait WorkflowInterceptorLogic {
  def taskPersistence: TaskPersistence

  def updateWorkflowProperties(release: Release): Unit = {
    if (release.isWorkflow) {
      // disable notification silently for workflows
      release.setDisableNotifications(true)
    }
    updateWorkflowTaskProperties(release)
  }

  def updateWorkflowProperties(task: Task): Unit = {
    val release = task.getRelease
    if (release.isWorkflow) {
      if (isTaskUpdatable(task)) {
        setNewOwnerAndTeam(task, task.getReleaseOwner)
      }
    }
  }

  private def updateWorkflowTaskProperties(release: Release): Unit = {
    val tasksToUpdate = if (release.isWorkflow) {
      for {
        task <- release.getAllTasks.asScala
        if shouldTaskBeUpdated(release, task)
      } yield {
        setNewOwnerAndTeam(task, release.getOwner)
        resetDueSoonAndOverdueNotifiedFlags(task)
        task
      }
    } else {
      Seq.empty
    }
    // when release is intercepted it will not save ALL tasks,
    //  only newly added ones will be saved so we have to update task properties manually
    // we must use taskPersistence instead of taskRepository in order to avoid extra intercept method calls
    taskPersistence.batchUpdateTaskProperties(tasksToUpdate.toSet)
  }

  private def shouldTaskBeUpdated(release: Release, task: Task): Boolean = {
    isTaskUpdatable(task) && release.getOwner != task.getOwner || task.getTeam != null
  }

  private def isTaskUpdatable(task: Task): Boolean = {
    task.isPlanned || task.isActive || task.isDoneInAdvance
  }

  private def setNewOwnerAndTeam(task: Task, newOwner: String): Unit = {
    task.setOwner(newOwner)
    task.setTeam(null)
  }

  private def resetDueSoonAndOverdueNotifiedFlags(task: Task): Unit = {
    task.setDueSoonNotified(false)
    task.setOverdueNotified(false)
  }
}
