package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlplatform.coc.dto.SCMTraceabilityData
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.SCM_DATA
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.Date

@Repository
@IsTransactional
class SCMTraceabilityPersistence @Autowired()(
                                               @Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                               @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect
                                             ) extends PersistenceSupport with Logging {

  private def obtainDate(scmData: SCMTraceabilityData): Date = Option(scmData.getDate).map(_.toDate).orNull

  private val STMT_FIND_BY_ID =
    s"""|SELECT *
        |FROM ${SCM_DATA.TABLE} scm
        |WHERE scm.${SCM_DATA.SCM_DATA_ID} = :id
     """.stripMargin

  def findById(scmId: Integer): Option[SCMTraceabilityData] = {
    sqlQuery(STMT_FIND_BY_ID,
      params("id" -> scmId),
      scmTraceabilityRowMapper).headOption
  }

  private val STMT_INSERT_INTO =
    s"""|INSERT INTO ${SCM_DATA.TABLE}
        |(${SCM_DATA.SCM_KIND}
        | ,${SCM_DATA.SCM_COMMIT}
        | ,${SCM_DATA.SCM_AUTHOR}
        | ,${SCM_DATA.SCM_DATE}
        | ,${SCM_DATA.SCM_MESSAGE}
        | ,${SCM_DATA.SCM_REMOTE}
        | ,${SCM_DATA.SCM_FILENAME})
        |VALUES
        |(
        | :kind
        | , :commit
        | , :author
        | , :date
        | , :message
        | , :remote
        | , :filename
        |)
     """.stripMargin

  def insert(data: SCMTraceabilityData): Integer = {
    sqlInsert(
      pkName(SCM_DATA.SCM_DATA_ID),
      STMT_INSERT_INTO,
      params(
        "kind" -> data.getKind,
        "commit" -> data.getCommit,
        "author" -> data.getAuthor,
        "date" -> obtainDate(data),
        "message" -> data.getMessage,
        "remote" -> data.getRemote,
        "filename" -> data.getFileName
      )
    )
  }

  private val STMT_UPDATE_SCM_DATA =
    s"""
       | UPDATE ${SCM_DATA.TABLE}
       | SET ${SCM_DATA.SCM_KIND} = :newSCMKind,
       | ${SCM_DATA.SCM_COMMIT} = :newSCMCommit,
       | ${SCM_DATA.SCM_AUTHOR} = :newSCMAuthor,
       | ${SCM_DATA.SCM_DATE} = :newSCMDate,
       | ${SCM_DATA.SCM_MESSAGE} = :newSCMMessage,
       | ${SCM_DATA.SCM_REMOTE} = :newSCMRemote,
       | ${SCM_DATA.SCM_FILENAME} = :newSCMFilename
       | WHERE
       | ${SCM_DATA.SCM_DATA_ID} = :scmId
    """.stripMargin

  def update(id: CiUid, scmEntity: SCMTraceabilityData): Unit = {
    sqlUpdate(STMT_UPDATE_SCM_DATA, params(
      "newSCMKind" -> scmEntity.getKind,
      "newSCMCommit" -> scmEntity.getCommit,
      "newSCMAuthor" -> scmEntity.getAuthor,
      "newSCMDate" -> obtainDate(scmEntity),
      "newSCMMessage" -> scmEntity.getMessage,
      "newSCMRemote" -> scmEntity.getRemote,
      "newSCMFilename" -> scmEntity.getFileName,
      "scmId" -> id
    ), () => _)
  }

  private val STMT_DELETE_SCM_DATA =
    s"""
       |DELETE FROM ${SCM_DATA.TABLE}
       |WHERE ${SCM_DATA.SCM_DATA_ID} = :scmId
     """.stripMargin

  def delete(id: CiUid): Boolean = {
    sqlExec(STMT_DELETE_SCM_DATA, params("scmId" -> id), _.execute())
  }

  val scmTraceabilityRowMapper: RowMapper[SCMTraceabilityData] = (rs: ResultSet, _: Int) => {
    val kind = rs.getString(SCM_DATA.SCM_KIND)
    val commit = rs.getString(SCM_DATA.SCM_COMMIT)
    val author = rs.getString(SCM_DATA.SCM_AUTHOR)
    val date = rs.getTimestamp(SCM_DATA.SCM_DATE)
    val message = rs.getString(SCM_DATA.SCM_MESSAGE)
    val remote = rs.getString(SCM_DATA.SCM_REMOTE)
    val fileName = rs.getString(SCM_DATA.SCM_FILENAME)

    new SCMTraceabilityData(kind, commit, author, new DateTime(date), message, remote, fileName)
  }
}
