package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlrelease.domain.{Dependency, Release}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, DependencyPersistence}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions

trait BaseReleaseItemRepository {

  def repositoryAdapter: SqlRepositoryAdapter

  def dependencyPersistence: DependencyPersistence

  def getRelease(releaseId: String, entityId: String): Release = {
    getRelease(releaseId, entityId, ResolveOptions.WITH_DECORATORS)
  }

  def getRelease(releaseId: String, entityId: String, resolveOptions: ResolveOptions): Release = {
    val release = repositoryAdapter.read[Release](releaseId)
    if (null == release) {
      throw new NotFoundException(s"Repository entity [$entityId] not found")
    }
    release
  }

  def notNull[T](ci: T, ciId: String): T = {
    if (ci == null) {
      throw new NotFoundException(s"Repository entity [$ciId] not found")
    } else {
      ci
    }
  }

  def insertDependencyWithGateTaskUid(gateTaskUid: CiUid)(dependency: Dependency): Unit = {
    if (!dependency.isArchived) {
      dependencyPersistence.insertDependencyWithGateTaskUid(gateTaskUid)(dependency)
    }
  }

  def insertDependency(dependency: Dependency): Unit = {
    if (!dependency.isArchived) {
      dependencyPersistence.insertDependency(dependency)
    }
  }

  def checkIsNotReferencedByDependencies(planItemIdOrItsChildren: CiId): Unit = {
    val externalIncomingDependencies = dependencyPersistence.findByPartialTargetIds(Set(planItemIdOrItsChildren), None)
      .map(_.fullDependencyId)
      .filterNot(_.startsWith(planItemIdOrItsChildren.normalized))
    if (externalIncomingDependencies.nonEmpty) {
      throw new ItemInUseException(s"Cannot delete [$planItemIdOrItsChildren] because it or one of its children is referenced by " +
        s"one or more dependencies: [${externalIncomingDependencies.mkString(", ")}]")
    }
  }

}
