package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.Team.{RELEASE_ADMIN_TEAMNAME, TEMPLATE_OWNER_TEAMNAME}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.{ACTIVE_STATUSES, PLANNED, TEMPLATE}
import com.xebialabs.xlrelease.repository.TeamRepository
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow.Root
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.upgrade.common.BaseInitializingUpgrade
import com.xebialabs.xlrelease.utils.FolderId
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

trait BaseAdminTeamsPermissionsUpgrade extends BaseInitializingUpgrade
    with PersistenceSupport
    with Logging {

  val teamRepository: TeamRepository

  protected def upgradeCiTeams(ci: SecuredCi, releasePermission: Permission, templatePermission: Permission): Unit = {
    upgradeCiTeams(ci, Seq(releasePermission), Seq(templatePermission))
  }

  protected def upgradeCiTeams(ci: SecuredCi, releasePermissions: Seq[Permission], templatePermissions: Seq[Permission]): Unit = {
    logger.trace(s"Upgrading teams of ${ci.getId}")
    val teams = teamRepository.getTeams(ci).asScala.toSeq
    if (teams.nonEmpty) {
      upgradeTeam(teams, RELEASE_ADMIN_TEAMNAME, releasePermissions)
      upgradeTeam(teams, TEMPLATE_OWNER_TEAMNAME, templatePermissions)
      teamRepository.saveTeamsToPlatform(ci.getId, teams.asJava)
    }
  }

  protected def upgradeTeam(teams: Seq[Team], teamName: String, permissions: Seq[Permission]): Unit = {
    teams.find(t => teamName == t.getTeamName).foreach { t =>
      logger.trace(s"Granting ${permissions.map(_.getPermissionName).mkString(", ")} to team ${t.getId}")
      t.addPermissions(permissions.map(_.getPermissionName).toArray)
    }
  }

  private val STMT_FIND_RELEASE_SECURED_CIS =
    s"""
       |SELECT
       |  r.${RELEASES.RELEASE_ID},
       |  r.${RELEASES.CI_UID}
       |FROM
       |  ${RELEASES.TABLE} r
       |WHERE
       |  r.${RELEASES.STATUS} IN (:statuses) AND
       |  r.${RELEASES.CI_UID} = r.${RELEASES.SECURITY_UID}
     """.stripMargin

  protected def findReleaseSecuredCis: Seq[SecuredCi] = {
    val statuses = (ACTIVE_STATUSES :+ PLANNED :+ TEMPLATE).toSeq.map(_.value()).asJava
    sqlQuery(STMT_FIND_RELEASE_SECURED_CIS, params("statuses" -> statuses), rs => {
      new SecuredCi(rs.getString(RELEASES.RELEASE_ID), rs.getInt(RELEASES.CI_UID))
    }).toSeq
  }

  private val STMT_FIND_FOLDER_SECURED_CIS =
    s"""
       |SELECT
       |  f.${FOLDERS.FOLDER_PATH},
       |  f.${FOLDERS.FOLDER_ID},
       |  f.${FOLDERS.CI_UID}
       |FROM
       |  ${FOLDERS.TABLE} f
       |WHERE
       |  f.${FOLDERS.CI_UID} = f.${FOLDERS.SECURITY_UID} AND
       |  f.${FOLDERS.CI_UID} <> ${Root.uid}
     """.stripMargin

  protected def findFolderSecuredCis: Seq[SecuredCi] = {
    sqlQuery(STMT_FIND_FOLDER_SECURED_CIS, params(), rs => {
      val folderId = FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)
      new SecuredCi(folderId.absolute, rs.getInt(FOLDERS.CI_UID))
    }).toSeq
  }
}
