package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.calendar.CalendarEntry
import com.xebialabs.xlrelease.repository.CalendarEntryRepository
import com.xebialabs.xlrelease.repository.sql.SqlCalendarEntryRepository._
import com.xebialabs.xlrelease.repository.sql.persistence.CalendarEntryPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.data.CalendarEntryRow
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.utils.TypeHelper

import java.util
import java.util.Date
import scala.jdk.CollectionConverters._

@IsTransactional
class SqlCalendarEntryRepository(calendarEntryPersistence: CalendarEntryPersistence,
                                 sqlRepositoryAdapter: SqlRepositoryAdapter) extends CalendarEntryRepository {

  override def create(calendarEntry: CalendarEntry): Unit = {
    calendarEntryPersistence.insert(calendarEntry.asRow)
  }

  override def update(calendarEntry: CalendarEntry): Unit = {
    calendarEntryPersistence.update(calendarEntry.asRow)
  }

  override def delete(id: String): Unit = {
    calendarEntryPersistence.deleteById(id)
  }

  override def exists(id: String): Boolean = calendarEntryPersistence.exists(id)

  override def findById[T <: CalendarEntry](id: String): T = {
    val row = calendarEntryPersistence.findById(id)
    deserialize[T](row.content)
  }

  override def findAllByType[T <: CalendarEntry](ciType: Type): util.List[T] = findAllByTypeInRange[T](ciType, null, null)

  override def findAllByTypeInRange[T <: CalendarEntry](ciType: Type, from: Date, to: Date): util.List[T] = {
    calendarEntryPersistence.search(TypeHelper.getAllSubtypesOf(ciType), from, to).map(row => deserialize[T](row.content)).toList.asJava
  }

  @IsReadOnly
  override def existsByTypeInRange(ciType: Type, from: Date, to: Date): Boolean = {
    calendarEntryPersistence.existsByTypeInRange(TypeHelper.getAllSubtypesOf(ciType), from, to)
  }

  private def deserialize[T <: CalendarEntry](content: String): T = {
    // S-91304 there is no need for sqlRepositoryAdapter since calendar entries do not have references!?
    CiSerializerHelper.deserialize(content, sqlRepositoryAdapter).asInstanceOf[T]
  }

}

object SqlCalendarEntryRepository {

  implicit class CalendarEntryOps(val entry: CalendarEntry) extends AnyVal {
    def asRow: CalendarEntryRow = CalendarEntryRow(
      entry.getId,
      entry.getType,
      entry.getLabel,
      entry.getStartDate,
      entry.getEndDate,
      CiSerializerHelper.serialize(entry)
    )
  }

}
