package com.xebialabs.xlrelease.repository.sql.persistence

import com.github.benmanes.caffeine.cache.Cache
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.data.ReleaseRow
import com.xebialabs.xlrelease.support.cache.caffeine.spring.XlrCaffeineCacheManager
import grizzled.slf4j.Logging
import org.springframework.jmx.`export`.annotation.{ManagedOperation, ManagedResource}

import java.util.concurrent.ConcurrentHashMap

@ManagedResource(objectName = "com.xebialabs.xlrelease.settings:name=ReleaseCacheService", description = "MBean to configure internal release cache")
sealed trait ReleaseCacheServiceMBean {

  @ManagedOperation(description = "Invalidate provided entry with provided Release ID as a key")
  def invalidate(releaseId: CiId): Unit

  @ManagedOperation(description = "Invalidate all entries")
  def invalidateAll(): Unit

  @ManagedOperation(description = "Get cache statistics as a string")
  def getStats(): String

}

class ReleaseCacheService(cacheManager: XlrCaffeineCacheManager) extends ReleaseCacheServiceMBean with Logging {

  private val allowedReleases = new ConcurrentHashMap[String, String]()
  private val cacheName = "release-rows"

  def getCache: Cache[CiId, ReleaseRow] = {
    cacheManager.getOrCreateCache(cacheName)
  }

  def get(releaseId: CiId): Option[ReleaseRow] = {
    Option(getCache.getIfPresent(releaseId.shortId)).map { r: ReleaseRow =>
      logger.trace(s"Fetching ${releaseId.shortId} from cache")
      r
    }
  }

  def put(releaseId: CiId, releaseRow: ReleaseRow): Unit = {
    if (allowedReleases.containsKey(releaseId.shortId)) {
      logger.trace(s"Caching ${releaseId.shortId}")
      getCache.put(releaseId.shortId, releaseRow)
    }
  }

  def allow(releaseId: CiId): Unit = {
    allowedReleases.putIfAbsent(releaseId.shortId, releaseId)
  }

  def disallow(releaseId: CiId): Unit = {
    allowedReleases.remove(releaseId.shortId)
    invalidate(releaseId)
  }

  override def invalidate(releaseId: CiId): Unit = {
    getCache.invalidate(releaseId.shortId)
  }

  override def invalidateAll(): Unit = {
    getCache.invalidateAll()
  }

  override def getStats(): String = {
    val cacheStats = getCache.stats()
    cacheStats.toString
  }
}
