package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.domain.utils.syntax._
import com.xebialabs.xlrelease.domain.{Dependency, Release, Task}
import com.xebialabs.xlrelease.repository.sql.persistence.CiUid
import com.xebialabs.xlrelease.repository.sql.persistence.CommentPersistence.CommentRow
import com.xebialabs.xlrelease.repository.{CiCloneHelper, Ids}
import com.xebialabs.xlrelease.utils.Diff
import org.joda.time.DateTime

import scala.jdk.CollectionConverters._

trait SqlReleaseRepositoryHelper {
  self: SqlReleaseRepository =>

  protected def replacePhases(original: Release, updated: Release): Unit = {
    val phasesDiff = Diff(
      original.getPhases.asScala,
      updated.getPhases.asScala
    )
    if (phasesDiff.newValues.nonEmpty) {
      phasePersistence.batchInsert(phasesDiff.newValues.toSet, original.getCiUid)
    }
    phasesDiff.deletedValues.foreach { phase =>
      phasePersistence.delete(phase)
    }
    phasesDiff.updatedValues.foreach { phase =>
      phasePersistence.update(phase)
    }
  }

  protected def replaceTasks(original: Release, updated: Release): Unit = {
    val tasksDiff = Diff(
      original.getAllTasks.asScala,
      updated.getAllTasks.asScala
    )
    tasksDiff.newValues.foreach { task =>
      val taskUid = taskPersistence.insert(task, original.getCiUid)
      task.dependencies.filterNot(_.isArchived).foreach(insertDependency(taskUid))

      task.comments.foreach(c => commentPersistence.create(taskUid, CommentRow.fromComment(c)))
    }
    tasksDiff.deletedValues.foreach { task =>
      task.dependencies.foreach(dependencyPersistence.deleteDependency)
      val taskUid = taskPersistence.taskUidById(task.getId).get

      commentPersistence.deleteAll(taskUid)
      taskPersistence.delete(task)
    }
    tasksDiff.updatedPairs.foreach { case (oTask, uTask) =>
      taskPersistence.updateProperties(uTask)
      uTask.dependencies.foreach(dependencyPersistence.updateDependency)
      updateCommentsOnTasks(oTask, uTask)
    }
  }

  protected def replaceReleaseExtensions(original: Release, updated: Release): Unit = {
    Diff(
      original.getExtensions.asScala.map(ext => ext.getId -> ext).toMap,
      updated.getExtensions.asScala.map(ext => ext.getId -> ext).toMap
    ).foreachValue(
      ext => releaseExtensionsRepository.create(ext),
      ext => releaseExtensionsRepository.update(ext),
      ext => releaseExtensionsRepository.delete(ext.getId)
    )
  }

  protected def replaceReleaseTeams(original: Release, updated: Release): Unit = {
    if (Ids.isInRootFolder(original.getId)) {
      val teamDiff = Diff(
        original.getTeams.asScala,
        updated.getTeams.asScala
      )
      teamDiff.deletedValues.foreach(team => teamRepository.delete(team.getId))
      teamDiff.updatedValues.foreach(teamRepository.update)
      teamDiff.newValues.foreach(teamRepository.create(updated.getId, _))
    }
  }

  protected def replaceTemplateLogo(original: Release, updated: Release): Unit = {
    val updatedTemplateLogo = updated.getLogo
    releasePersistence.fixLogoId(updated.getId, updatedTemplateLogo)
    releasePersistence.updateLogo(updated.getCiUid, Option(original.getLogo), updatedTemplateLogo)
  }

  protected def updateCommentsOnTasks(originalTask: Task, updatedTask: Task): Unit = {
    val taskUid = taskPersistence.taskUidById(updatedTask.getId).get
    val originalComments = CiCloneHelper.cloneCis(originalTask.getComments).asScala
    originalComments.foreach(comment => comment.setId(Ids.getName(comment.getId)))
    val updatedComments = CiCloneHelper.cloneCis(updatedTask.getComments).asScala
    updatedComments.foreach(comment => comment.setId(Ids.getName(comment.getId)))
    val commentsDiff = Diff(originalComments, updatedComments)

    commentsDiff.newValues.foreach { newComment =>
      commentPersistence.create(taskUid, CommentRow.fromComment(newComment))
    }
    commentsDiff.updatedPairs.withFilter { case (oComment, uComment) =>
      oComment.getText != uComment.getText || oComment.getDate != uComment.getDate
    }.foreach { case (_, uComment) =>
      commentPersistence.update(taskUid,
        uComment.getId,
        uComment.getText,
        Some(new DateTime(uComment.getDate)))
    }
    commentsDiff.deletedValues.foreach { deletedComment =>
      commentPersistence.delete(taskUid, deletedComment.getId)
    }
  }

  protected def insertDependency(gateTaskUid: CiUid)(dependency: Dependency): Unit = {
    dependencyPersistence.insertDependencyWithGateTaskUid(gateTaskUid)(dependency)
  }

}
