package com.xebialabs.xlrelease.security.sql.snapshots.persistence

import com.xebialabs.xlplatform.utils.ResourceManagement
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.json.JsonUtils
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.PERMISSION_SNAPSHOTS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import com.xebialabs.xlrelease.security.sql.snapshots.domain.RolesSnapshot
import com.xebialabs.xlrelease.security.sql.snapshots.persistence.SqlRolesSnapshotPersistence._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Service

import java.util.Date


object SqlRolesSnapshotPersistence {
  val INSERT_SNAPSHOT_QUERY =
    s"""INSERT INTO ${PERMISSION_SNAPSHOTS.TABLE} (
       |  ${PERMISSION_SNAPSHOTS.CONTAINER_ID},
       |  ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE},
       |  ${PERMISSION_SNAPSHOTS.CONTENT}
       |  ) VALUES (
       |  :${PERMISSION_SNAPSHOTS.CONTAINER_ID},
       |  :${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE},
       |  :${PERMISSION_SNAPSHOTS.CONTENT}
       |  )
       |  """.stripMargin

  val FIND_GLOBAL_SNAPSHOT_QUERY =
    s"""SELECT
       |  ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE},
       |  ${PERMISSION_SNAPSHOTS.CONTENT}
       |FROM ${PERMISSION_SNAPSHOTS.TABLE}
       |WHERE ${PERMISSION_SNAPSHOTS.CONTAINER_ID} IS NULL
       |AND ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} <= :SnapshotDate
       |ORDER BY ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} DESC
       |""".stripMargin

  val FIND_FOLDER_SNAPSHOT_QUERY =
    s"""SELECT
       |  ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE},
       |  ${PERMISSION_SNAPSHOTS.CONTENT}
       |FROM ${PERMISSION_SNAPSHOTS.TABLE}
       |WHERE ${PERMISSION_SNAPSHOTS.CONTAINER_ID} = :ContainerId
       |AND ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} <= :SnapshotDate
       |ORDER BY ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} DESC
       |""".stripMargin

  val FIND_GLOBAL_SNAPSHOTS_BY_START_END_DATE_QUERY =
    s"""SELECT
       |  ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE},
       |  ${PERMISSION_SNAPSHOTS.CONTENT}
       |FROM ${PERMISSION_SNAPSHOTS.TABLE}
       |WHERE ${PERMISSION_SNAPSHOTS.CONTAINER_ID} IS NULL
       |AND ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} >= :SnapshotStartDate
       |AND ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} <= :SnapshotEndDate
       |ORDER BY ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} ASC
       |""".stripMargin

  val FIND_FOLDER_SNAPSHOTS_BY_START_END_DATE_QUERY =
    s"""SELECT
       |  ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE},
       |  ${PERMISSION_SNAPSHOTS.CONTENT}
       |FROM ${PERMISSION_SNAPSHOTS.TABLE}
       |WHERE ${PERMISSION_SNAPSHOTS.CONTAINER_ID} = :ContainerId
       |AND ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} >= :SnapshotStartDate
       |AND ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} <= :SnapshotEndDate
       |ORDER BY ${PERMISSION_SNAPSHOTS.SNAPSHOT_DATE} ASC
       |""".stripMargin
}

@Service
class SqlRolesSnapshotPersistence(@Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                  @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect) extends SqlRepository with PersistenceSupport
  with LimitOffset with Logging with Utils {

  def saveSnapshot(snapshot: RolesSnapshot): Unit = {
    val json = serializeSnapshot(snapshot)
    sqlExecWithContent(
      INSERT_SNAPSHOT_QUERY,
      params(
        PERMISSION_SNAPSHOTS.CONTAINER_ID -> snapshot.containerIdName.orNull,
        PERMISSION_SNAPSHOTS.SNAPSHOT_DATE -> snapshot.snapshotDate.asTimestamp
      ),
      PERMISSION_SNAPSHOTS.CONTENT -> json,
      _ => ())
  }

  def findFirstSnapshot(containerId: Option[String], byDate: Date): Option[RolesSnapshot] = {
    if (containerId.isDefined) {
      findOne {
        sqlQuery(
          addLimitAndOffset(
            FIND_FOLDER_SNAPSHOT_QUERY,
            Some(1)),
          params(
            "ContainerId" -> containerId.map(Ids.getName).get,
            "SnapshotDate" -> byDate.asTimestamp
          ),
          snapshotRowMapper
        )
      }
    } else {
      findOne {
        sqlQuery(
          addLimitAndOffset(
            FIND_GLOBAL_SNAPSHOT_QUERY,
            Some(1)),
          params(
            "SnapshotDate" -> byDate.asTimestamp
          ),
          snapshotRowMapper
        )
      }
    }
  }

  def findSnapshots(containerId: Option[String], startDate: Date, endDate: Date): Seq[RolesSnapshot] = {
    if (containerId.isDefined) {
      findMany {
        sqlQuery(
          FIND_FOLDER_SNAPSHOTS_BY_START_END_DATE_QUERY,
          params(
            "ContainerId" -> containerId.map(Ids.getName).get,
            "SnapshotStartDate" -> startDate.asTimestamp,
            "SnapshotEndDate" -> endDate.asTimestamp
          ),
          snapshotRowMapper
        )
      }
    } else {
      findMany {
        sqlQuery(
          FIND_GLOBAL_SNAPSHOTS_BY_START_END_DATE_QUERY,
          params(
            "SnapshotStartDate" -> startDate.asTimestamp,
            "SnapshotEndDate" -> endDate.asTimestamp
          ),
          snapshotRowMapper
        )
      }
    }
  }

  private def serializeSnapshot(snapshot: RolesSnapshot): String = {
    JsonUtils.objectMapper.writeValueAsString(snapshot)
  }

  private val snapshotRowMapper: RowMapper[RolesSnapshot] = (rs, _) => {
    val content = ResourceManagement.using(rs.getBinaryStream(PERMISSION_SNAPSHOTS.CONTENT))(decompress)
    val snapshot = JsonUtils.objectMapper.readValue(content, classOf[RolesSnapshot])
    val timestamp = rs.getTimestamp(PERMISSION_SNAPSHOTS.SNAPSHOT_DATE)
    RolesSnapshot(
      timestamp,
      snapshot.containerIdName,
      snapshot.rolesAndPermissions,
      snapshot.isInherited,
      snapshot.parentContainerIdName
    )
  }

}
