package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.facet.{Facet, FacetScope, TaskReportingRecord}
import com.xebialabs.xlrelease.domain.id.{CiUid, IdCreator}
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, RichCiId}
import com.xebialabs.xlrelease.repository.sql.persistence.FacetPersistence._
import com.xebialabs.xlrelease.repository.sql.persistence.FacetSqlBuilder.STMT_SELECT_FACET
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FACETS, FOLDERS, TASKS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.data.FacetRow
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

object FacetPersistence {
  def extractRetryAttemptNumber(facet: Facet): Int = {
    Option(facet)
      .collect { case f: TaskReportingRecord => f.getRetryAttemptNumber }
      .getOrElse(0)
  }

  def extractCreatedViaApi(facet: Facet): Integer = {
    Option(facet)
      .collect { case f: TaskReportingRecord => f.isCreatedViaApi }
      .getOrElse(false) match {
      case true => 1
      case false => 0
    }
  }
}

@IsTransactional
class FacetPersistence(taskPersistence: TaskPersistence,
                       implicit val jdbcTemplate: JdbcTemplate,
                       implicit val dialect: Dialect)
  extends SqlRepository with PersistenceSupport {

  private val STMT_GET_FACET_UID_BY_ID = s"SELECT ${FACETS.CI_UID} FROM ${FACETS.TABLE} WHERE ${FACETS.FACET_ID} = :facetId"

  def getFacetUid(facetId: CiId): CiUid =
    sqlQuery(STMT_GET_FACET_UID_BY_ID, params("facetId" -> getName(facetId.normalized)), rs => CiUid(rs.getLong(FACETS.CI_UID))).head

  private val STMT_INSERT_FACET =
    s"""
       |INSERT INTO ${FACETS.TABLE}
       |  ( ${FACETS.CI_UID}
       |  ,${FACETS.FACET_ID}
       |  , ${FACETS.CI_TYPE}
       |  , ${FACETS.APPLIED_TO_TASK_UID}
       |  , ${FACETS.CONTENT}
       |  , ${FACETS.RETRY_ATTEMPT_NUMBER}
       |  , ${FACETS.CREATED_VIA_API}
       | )
       | VALUES
       | ( :ciUid
       | , :facetId
       | , :ciType
       | , :appliedToTaskUid
       | , :content
       | , :retryAttemptNumber
       | , :createdViaApi
       | )
     """.stripMargin

  def create(facet: Facet): Facet = {
    val content = serialize(facet)
    try {
      val ciUid = IdCreator.generateId
      sqlExecWithContent(STMT_INSERT_FACET, params(
        "ciUid" -> ciUid,
        "facetId" -> getName(facet.getId.normalized),
        "ciType" -> facet.getType.toString,
        "appliedToTaskUid" -> (if (facet.getScope == FacetScope.TASK) taskPersistence.getTaskUidById(facet.getTargetId).get else null),
        "retryAttemptNumber" -> extractRetryAttemptNumber(facet),
        "createdViaApi" -> extractCreatedViaApi(facet)
      ), "content" -> content)
      facet
    } catch {
      case ex: DuplicateKeyException => throw new IllegalArgumentException(s"Facet with ID '${facet.getId}' already exists", ex)
    }
  }

  private val STMT_DELETE_FACET =
    s"""
       | DELETE FROM ${FACETS.TABLE}
       | WHERE ${FACETS.FACET_ID} = :facetId
     """.stripMargin

  def delete(facetId: CiId): Unit =
    sqlUpdate(STMT_DELETE_FACET, params("facetId" -> getName(facetId.normalized)), _ => ())

  private val STMT_UPDATE_FACET =
    s"""
       | UPDATE ${FACETS.TABLE}
       | SET
       |   ${FACETS.CI_TYPE} = :type,
       |   ${FACETS.APPLIED_TO_TASK_UID} = :appliedToTaskUid,
       |   ${FACETS.CONTENT} = :content,
       |   ${FACETS.RETRY_ATTEMPT_NUMBER} = :retryAttemptNumber,
       |   ${FACETS.CREATED_VIA_API} = :createdViaApi
       | WHERE
       | ${FACETS.FACET_ID} = :facetId
     """.stripMargin

  def update(facet: Facet): Facet = {
    val content = serialize(facet)
    sqlExecWithContent(STMT_UPDATE_FACET, params(
      "type" -> facet.getType.toString,
      "appliedToTaskUid" -> (if (facet.getScope == FacetScope.TASK) taskPersistence.getTaskUidById(facet.getTargetId).get else null),
      "facetId" -> getName(facet.getId.normalized),
      "retryAttemptNumber" -> extractRetryAttemptNumber(facet),
      "createdViaApi" -> extractCreatedViaApi(facet)
    ), "content" -> content,
      checkCiUpdated(facet.getId)
    )
    facet
  }

  // TODO: replace with EXISTS statement, why do we use count?
  private val STMT_EXISTS_FACET_BY_ID: String = s"SELECT COUNT(*) FROM ${FACETS.TABLE} WHERE ${FACETS.FACET_ID} = :facetId"

  def exists(facetId: CiId): Boolean = sqlQuery(STMT_EXISTS_FACET_BY_ID, params("facetId" -> getName(facetId.normalized)), _.getInt(1) > 0).head

  private val STMT_GET_FACET =
    s"""$STMT_SELECT_FACET
       | WHERE
       |  ${FACETS.FACET_ID} = :facetId
     """.stripMargin

  def get(facetId: CiId): Option[FacetRow] =
    sqlQuery(STMT_GET_FACET, params("facetId" -> getName(facetId.normalized)), facetRowMapper).headOption

  def findByParentId(parentId: String): Seq[FacetRow] = search(new FacetSqlBuilder()
    .select
    .withParentId(parentId)
    .build()
  )

  def findByTargetId(targetId: String): Seq[FacetRow] = search(new FacetSqlBuilder()
    .select
    .withTargetId(targetId)
    .build()
  )

  def search(sqlWithParameters: SqlWithParameters): Seq[FacetRow] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query[FacetRow](sql, facetRowMapper, params: _*).asScala.toSeq
  }

  private val facetRowMapper: RowMapper[FacetRow] = (rs: ResultSet, _: Int) => {
    using(rs.getBinaryStream(FACETS.CONTENT)) { contentStream =>
      val content = decompress(contentStream)
      FacetRow(
        content,
        rs.getString(FACETS.FACET_ID),
        rs.getLong(FACETS.CI_UID),
        rs.getString(FACETS.CI_TYPE),
        (FolderId.Root / rs.getString(FOLDERS.FOLDER_PATH) / rs.getString(FOLDERS.FOLDER_ID) / rs.getString(TASKS.TASK_ID)).absolute
      )
    }
  }
}
