package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TASK_BACKUPS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet

@IsTransactional
class TaskBackupPersistence(implicit val jdbcTemplate: JdbcTemplate, implicit val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with Utils with Logging {

  def upsertTaskBackup(task: Task): Unit = {
    findTaskBackupId(task.getId, task.getReleaseUid) match {
      case Some(_) => updateTaskBackupJson(task)
      case None => insertTaskBackupJson(task)
    }
  }

  def updateTaskBackupJson(task: Task): Unit = {
    val taskId: String = task.getId
    logger.debug(s"Updating existing task $taskId backup into database")
    sqlExecWithContent(
      s"""UPDATE ${TASK_BACKUPS.TABLE}
         | SET ${TASK_BACKUPS.CONTENT} = :content
         | WHERE
         | ${TASK_BACKUPS.TASK_ID} = :taskId AND
         | ${TASK_BACKUPS.CI_UID} = :releaseCiUid""".stripMargin,
      params(
        "releaseCiUid" -> task.getReleaseUid,
        "taskId" -> getReleaselessChildId(taskId)
      ), "content" -> serialize(task),
      _ => ()
    )
  }

  private def insertTaskBackupJson(task: Task): Unit = {
    val taskId: String = task.getId
    logger.debug(s"Inserting new task $taskId backup into database, releaseCiUid is ${task.getReleaseUid}")
    sqlExecWithContent(
      s"""INSERT INTO ${TASK_BACKUPS.TABLE} (
         |   ${TASK_BACKUPS.TASK_ID},
         |   ${TASK_BACKUPS.CI_UID},
         |   ${TASK_BACKUPS.CONTENT})
         | VALUES (
         |   :taskId,
         |   :releaseCiUid,
         |   :content)""".stripMargin,
      params(
        "releaseCiUid" -> task.getReleaseUid,
        "taskId" -> getReleaselessChildId(taskId)
      ), "content" -> serialize(task),
      _ => ()
    )
  }

  def findTaskBackup(taskId: String, releaseCiUid: CiUid): Option[String] = {
    val releaseId = releaseIdFrom(taskId)
    findOptional(_.queryForObject(
      s"""SELECT ${TASK_BACKUPS.CONTENT} FROM ${TASK_BACKUPS.TABLE}
         | WHERE
         | ${TASK_BACKUPS.TASK_ID} = ? AND
         | ${TASK_BACKUPS.CI_UID} = ?
       """.stripMargin,
      binaryStreamRowMapper,
      getReleaselessChildId(taskId),
      releaseCiUid)
    )
  }

  private def findTaskBackupId(taskId: String, releaseCiUid: CiUid): Option[String] = {
    val releaseId = releaseIdFrom(taskId)
    findOptional(_.queryForObject(
      s"""SELECT ${TASK_BACKUPS.TASK_ID} FROM ${TASK_BACKUPS.TABLE}
         | WHERE
         | ${TASK_BACKUPS.TASK_ID} = ? AND
         | ${TASK_BACKUPS.CI_UID} = ?
       """.stripMargin,
      classOf[String],
      getReleaselessChildId(taskId),
      releaseCiUid)
    )
  }

  private val binaryStreamRowMapper: RowMapper[String] = (rs: ResultSet, _: Int) => {
    val inputStream = rs.getBinaryStream(TASK_BACKUPS.CONTENT)
    try {
      decompress(inputStream)
    } finally {
      inputStream.close()
    }
  }
}

class TaskBackupStoreException(msg: String, cause: Throwable) extends RuntimeException(msg, cause) {
  def this(msg: String) = this(msg, null)
}
