package com.xebialabs.xlrelease.repository.sql.persistence.configuration

import com.xebialabs.xlplatform.utils.ResourceManagement
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{CONFIGURATIONS, ConfigurationRefs, FOLDERS}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, rowMapper}
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence.{ConfigurationRow, mapFolderPath}
import com.xebialabs.xlrelease.repository.sql.persistence.data.ConfigurationReferenceRow
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.RowMapper

import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.Try

private[configuration] abstract class ConfigurationReferencePersistenceCommon
  extends ConfigurationReferencePersistence with Logging {

  protected def table: ConfigurationRefs

  protected def STMT_GET_REFS: String

  private val STMT_DELETE_REFS_BY_CONFIGURATION_UIDS_BATCH_SIZE = 512

  protected def configurationReferenceRowMapper: RowMapper[ConfigurationReferenceRow]

  protected def configurationBinaryStreamRowMapper(ciUid: CiUid): RowMapper[ConfigurationRow] = (rs: ResultSet, _: Int) => {
    val rawConfiguration = ResourceManagement.using(rs.getBinaryStream(CONFIGURATIONS.CONTENT))(decompress)
    (mapFolderPath(rs), rawConfiguration, ciUid)
  }

  private lazy val STMT_INSERT_REFS: String =
    s"""
       |INSERT INTO ${table.TABLE} (
       |  ${table.CONFIGURATION_UID},
       |  ${table.USED_BY_UID}
       |) VALUES (
       |  :ciUid,
       |  :usedByCiUid
       |)
     """.stripMargin

  private lazy val STMT_DELETE_REFS: String =
    s"""
       |DELETE FROM ${table.TABLE}
       | WHERE
       |  ${table.CONFIGURATION_UID} = :ciUid
       |  AND ${table.USED_BY_UID} = :usedByCiUid
       |""".stripMargin

  private lazy val STMT_COUNT_REFS: String =
    s"SELECT COUNT(*) FROM ${table.TABLE} WHERE ${table.CONFIGURATION_UID} = :configurationCiUid"

  private lazy val STMT_GET_REFERENCES_BY_UID: String =
    s"""
       |SELECT conf.${CONFIGURATIONS.ID}
       |  FROM ${table.TABLE} refs
       |  JOIN ${CONFIGURATIONS.TABLE} conf on refs.${table.CONFIGURATION_UID} = conf.${CONFIGURATIONS.CI_UID}
       |  WHERE refs.${table.USED_BY_UID} = :usedByCiUid
     """.stripMargin

  private lazy val STMT_DELETE_USED_BY_REF: String =
    s"""|DELETE FROM ${table.TABLE}
        | WHERE ${table.USED_BY_UID} = :usedByCiUid
     """.stripMargin

  private lazy val STMT_DELETE_REFS_BY_CONFIGURATION_UID: String =
    s"""|DELETE FROM ${table.TABLE}
        | WHERE ${table.CONFIGURATION_UID} = :configurationUid
     """.stripMargin

  private lazy val STMT_ALL_CONFIGURATIONS_BY_UIDS: String =
    s"""|DELETE FROM ${table.TABLE}
        | WHERE ${table.CONFIGURATION_UID} IN (:configurationUids)
     """.stripMargin

  protected lazy val STMT_ALL_CONFIGURATIONS_BY_UID =
    s"""|SELECT
        |  conf.${CONFIGURATIONS.CONTENT},
        |  f.${FOLDERS.FOLDER_PATH},
        |  f.${FOLDERS.FOLDER_ID}
        | FROM ${table.TABLE} refs
        | JOIN ${CONFIGURATIONS.TABLE} conf on refs.${table.CONFIGURATION_UID} = conf.${CONFIGURATIONS.CI_UID}
        | LEFT JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = ${CONFIGURATIONS.FOLDER_CI_UID}
        | WHERE refs.${table.USED_BY_UID} = :uid""".stripMargin

  override def insertRefs(configurationIds: Set[CiId], usedBy: CiUid): Unit = if (configurationIds.nonEmpty) {
    sqlBatch(STMT_INSERT_REFS, configurationIds.map(id => Map("ciUid" -> getUid(id).get, "usedByCiUid" -> usedBy)))
  }

  override def deleteRefs(configurationIds: Set[CiId], usedBy: CiUid): Unit = if (configurationIds.nonEmpty) {
    sqlBatch(STMT_DELETE_REFS, configurationIds.map(id => Map("ciUid" -> getUid(id).get, "usedByCiUid" -> usedBy)))
  }

  override def isReferenced(configurationId: String): Boolean = {
    val maybeUid = getUid(configurationId)
    maybeUid.exists(uid => sqlQuery(STMT_COUNT_REFS, params("configurationCiUid" -> uid), _.getInt(1) > 0).head)
  }

  override def getReferencingEntities(configurationId: String): Seq[ConfigurationReferenceRow] = {
    val maybeUid = getUid(configurationId)
    maybeUid.map(configurationCiUid =>
      sqlQuery[ConfigurationReferenceRow](STMT_GET_REFS, params(
        "configurationCiUid" -> configurationCiUid
      ), configurationReferenceRowMapper).toSeq
    ).getOrElse(
      Seq.empty
    )
  }

  override def getReferencesByUid(usedByCiUid: CiUid): List[String] = {
    val mapper: RowMapper[String] = rowMapper { rs =>
      rs.getString(CONFIGURATIONS.ID)
    }
    sqlQuery(STMT_GET_REFERENCES_BY_UID, params("usedByCiUid" -> usedByCiUid), mapper).toList
  }

  override def deleteUsedByRef(usedByCiUid: CiUid): Unit = {
    sqlUpdate(STMT_DELETE_USED_BY_REF, params(
      "usedByCiUid" -> usedByCiUid
    ), _ => ())
  }

  override def deleteRefsByConfigurationUid(configurationUid: CiUid): Unit = {
    sqlUpdate(STMT_DELETE_REFS_BY_CONFIGURATION_UID, params(
      "configurationUid" -> configurationUid
    ), _ => ())
  }


  override def deleteRefsByConfigurationUids(configurationUids: Seq[CiUid]): Unit = {
    configurationUids.grouped(STMT_DELETE_REFS_BY_CONFIGURATION_UIDS_BATCH_SIZE).foreach(ids => {
      sqlUpdate(STMT_ALL_CONFIGURATIONS_BY_UIDS, params(
        "configurationUids" -> ids.asJava
      ), _ => ())
    })
  }

  override def findAllByUid(uid: CiUid): Seq[ConfigurationRow]

  override def deleteRefsByTypes(ciTypes: Seq[String]): Try[Boolean] = {
    val stmt =
      s"""|DELETE FROM ${table.TABLE}
          | WHERE ${table.CONFIGURATION_UID} IN
          |  (SELECT ${CONFIGURATIONS.CI_UID} FROM ${CONFIGURATIONS.TABLE}
          |  WHERE ${CONFIGURATIONS.CI_TYPE} in (:ciTypes))
      """.stripMargin

    sqlExec(stmt, params("ciTypes" -> ciTypes.asJava), ps => Try(ps.execute()))
  }
}
